"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaService = void 0;
const common_1 = require("@nestjs/common");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const editing_dto_1 = require("../dtos/editing.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
const editor_1 = require("../utils/editor");
const media_1 = require("../utils/media");
const mime_types_1 = require("../utils/mime-types");
const misc_1 = require("../utils/misc");
const transform_1 = require("../utils/transform");
let MediaService = class MediaService extends base_service_1.BaseService {
    videoInterfaces = { dri: [], mali: false };
    async onBootstrap() {
        const [dri, mali] = await Promise.all([this.getDevices(), this.hasMaliOpenCL()]);
        this.videoInterfaces = { dri, mali };
    }
    async handleQueueGenerateThumbnails({ force }) {
        let jobs = [];
        const queueAll = async () => {
            await this.jobRepository.queueAll(jobs);
            jobs = [];
        };
        for await (const asset of this.assetJobRepository.streamForThumbnailJob(!!force)) {
            const assetFiles = (0, asset_util_1.getAssetFiles)(asset.files);
            if (!assetFiles.previewFile || !assetFiles.thumbnailFile || !asset.thumbhash || force) {
                jobs.push({ name: enum_1.JobName.AssetGenerateThumbnails, data: { id: asset.id } });
            }
            if (asset.edits.length > 0 &&
                (!assetFiles.editedPreviewFile || !assetFiles.editedThumbnailFile || !assetFiles.editedFullsizeFile || force)) {
                jobs.push({ name: enum_1.JobName.AssetEditThumbnailGeneration, data: { id: asset.id } });
            }
            if (jobs.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await queueAll();
            }
        }
        await queueAll();
        const people = this.personRepository.getAll(force ? undefined : { thumbnailPath: '' });
        for await (const person of people) {
            if (!person.faceAssetId) {
                const face = await this.personRepository.getRandomFace(person.id);
                if (!face) {
                    continue;
                }
                await this.personRepository.update({ id: person.id, faceAssetId: face.id });
            }
            jobs.push({ name: enum_1.JobName.PersonGenerateThumbnail, data: { id: person.id } });
            if (jobs.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await queueAll();
            }
        }
        await queueAll();
        return enum_1.JobStatus.Success;
    }
    async handleQueueMigration() {
        const { active, waiting } = await this.jobRepository.getJobCounts(enum_1.QueueName.Migration);
        if (active === 1 && waiting === 0) {
            await this.storageCore.removeEmptyDirs(enum_1.StorageFolder.Thumbnails);
            await this.storageCore.removeEmptyDirs(enum_1.StorageFolder.EncodedVideo);
        }
        let jobs = [];
        const assets = this.assetJobRepository.streamForMigrationJob();
        for await (const asset of assets) {
            jobs.push({ name: enum_1.JobName.AssetFileMigration, data: { id: asset.id } });
            if (jobs.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await this.jobRepository.queueAll(jobs);
                jobs = [];
            }
        }
        await this.jobRepository.queueAll(jobs);
        jobs = [];
        for await (const person of this.personRepository.getAll()) {
            jobs.push({ name: enum_1.JobName.PersonFileMigration, data: { id: person.id } });
            if (jobs.length === constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await this.jobRepository.queueAll(jobs);
                jobs = [];
            }
        }
        await this.jobRepository.queueAll(jobs);
        return enum_1.JobStatus.Success;
    }
    async handleAssetMigration({ id }) {
        const { image } = await this.getConfig({ withCache: true });
        const asset = await this.assetJobRepository.getForMigrationJob(id);
        if (!asset) {
            return enum_1.JobStatus.Failed;
        }
        await this.storageCore.moveAssetImage(asset, enum_1.AssetFileType.FullSize, image.fullsize.format);
        await this.storageCore.moveAssetImage(asset, enum_1.AssetFileType.Preview, image.preview.format);
        await this.storageCore.moveAssetImage(asset, enum_1.AssetFileType.Thumbnail, image.thumbnail.format);
        await this.storageCore.moveAssetVideo(asset);
        return enum_1.JobStatus.Success;
    }
    async handleAssetEditThumbnailGeneration({ id }) {
        const asset = await this.assetJobRepository.getForGenerateThumbnailJob(id);
        const config = await this.getConfig({ withCache: true });
        if (!asset) {
            this.logger.warn(`Thumbnail generation failed for asset ${id}: not found in database or missing metadata`);
            return enum_1.JobStatus.Failed;
        }
        const generated = await this.generateEditedThumbnails(asset, config);
        await this.syncFiles(asset.files.filter((asset) => asset.isEdited), generated?.files ?? []);
        let thumbhash = generated?.thumbhash;
        if (!thumbhash) {
            const extractedImage = await this.extractOriginalImage(asset, config.image);
            const { info, data, colorspace } = extractedImage;
            thumbhash = await this.mediaRepository.generateThumbhash(data, {
                colorspace,
                processInvalidImages: false,
                raw: info,
                edits: [],
            });
        }
        if (!asset.thumbhash || Buffer.compare(asset.thumbhash, thumbhash) !== 0) {
            await this.assetRepository.update({ id: asset.id, thumbhash });
        }
        const fullsizeDimensions = generated?.fullsizeDimensions ?? (0, asset_util_1.getDimensions)(asset.exifInfo);
        await this.assetRepository.update({ id: asset.id, ...fullsizeDimensions });
        return enum_1.JobStatus.Success;
    }
    async handleGenerateThumbnails({ id }) {
        const asset = await this.assetJobRepository.getForGenerateThumbnailJob(id);
        const config = await this.getConfig({ withCache: true });
        if (!asset) {
            this.logger.warn(`Thumbnail generation failed for asset ${id}: not found in database or missing metadata`);
            return enum_1.JobStatus.Failed;
        }
        if (asset.visibility === enum_1.AssetVisibility.Hidden) {
            this.logger.verbose(`Thumbnail generation skipped for asset ${id}: not visible`);
            return enum_1.JobStatus.Skipped;
        }
        let generated;
        if (asset.type === enum_1.AssetType.Video || asset.originalFileName.toLowerCase().endsWith('.gif')) {
            this.logger.verbose(`Thumbnail generation for video ${id} ${asset.originalPath}`);
            generated = await this.generateVideoThumbnails(asset, config);
        }
        else if (asset.type === enum_1.AssetType.Image) {
            this.logger.verbose(`Thumbnail generation for image ${id} ${asset.originalPath}`);
            generated = await this.generateImageThumbnails(asset, config);
        }
        else {
            this.logger.warn(`Skipping thumbnail generation for asset ${id}: ${asset.type} is not an image or video`);
            return enum_1.JobStatus.Skipped;
        }
        const editedGenerated = await this.generateEditedThumbnails(asset, config);
        if (editedGenerated) {
            generated.files.push(...editedGenerated.files);
        }
        await this.syncFiles(asset.files, generated.files);
        const thumbhash = editedGenerated?.thumbhash || generated.thumbhash;
        if (!asset.thumbhash || Buffer.compare(asset.thumbhash, thumbhash) !== 0) {
            await this.assetRepository.update({ id: asset.id, thumbhash });
        }
        return enum_1.JobStatus.Success;
    }
    async extractImage(originalPath, minSize) {
        let extracted = await this.mediaRepository.extract(originalPath);
        if (extracted && !(await this.shouldUseExtractedImage(extracted.buffer, minSize))) {
            extracted = null;
        }
        return extracted;
    }
    async decodeImage(thumbSource, exifInfo, targetSize) {
        const { image } = await this.getConfig({ withCache: true });
        const colorspace = this.isSRGB(exifInfo) ? enum_1.Colorspace.Srgb : image.colorspace;
        const decodeOptions = {
            colorspace,
            processInvalidImages: process.env.IMMICH_PROCESS_INVALID_IMAGES === 'true',
            size: targetSize,
            orientation: exifInfo.orientation ? Number(exifInfo.orientation) : undefined,
        };
        const { info, data } = await this.mediaRepository.decodeImage(thumbSource, decodeOptions);
        return { info, data, colorspace };
    }
    async extractOriginalImage(asset, image, useEdits = false) {
        const extractEmbedded = image.extractEmbedded && mime_types_1.mimeTypes.isRaw(asset.originalFileName);
        const extracted = extractEmbedded ? await this.extractImage(asset.originalPath, image.preview.size) : null;
        const generateFullsize = ((image.fullsize.enabled || asset.exifInfo.projectionType === 'EQUIRECTANGULAR') &&
            !mime_types_1.mimeTypes.isWebSupportedImage(asset.originalPath)) ||
            useEdits;
        const convertFullsize = generateFullsize && (!extracted || !mime_types_1.mimeTypes.isWebSupportedImage(` .${extracted.format}`));
        const { data, info, colorspace } = await this.decodeImage(extracted ? extracted.buffer : asset.originalPath, extracted ? asset.exifInfo : { ...asset.exifInfo, orientation: null }, convertFullsize ? undefined : image.preview.size);
        return {
            extracted,
            data,
            info,
            colorspace,
            convertFullsize,
            generateFullsize,
        };
    }
    async generateImageThumbnails(asset, { image }, useEdits = false) {
        const previewFile = this.getImageFile(asset, {
            fileType: enum_1.AssetFileType.Preview,
            format: image.preview.format,
            isEdited: useEdits,
            isProgressive: !!image.preview.progressive && image.preview.format !== enum_1.ImageFormat.Webp,
        });
        const thumbnailFile = this.getImageFile(asset, {
            fileType: enum_1.AssetFileType.Thumbnail,
            format: image.thumbnail.format,
            isEdited: useEdits,
            isProgressive: !!image.thumbnail.progressive && image.thumbnail.format !== enum_1.ImageFormat.Webp,
        });
        this.storageCore.ensureFolders(previewFile.path);
        const extractedImage = await this.extractOriginalImage(asset, image, useEdits);
        const { info, data, colorspace, generateFullsize, convertFullsize, extracted } = extractedImage;
        const thumbnailOptions = { colorspace, processInvalidImages: false, raw: info, edits: useEdits ? asset.edits : [] };
        const promises = [
            this.mediaRepository.generateThumbhash(data, thumbnailOptions),
            this.mediaRepository.generateThumbnail(data, { ...image.thumbnail, ...thumbnailOptions }, thumbnailFile.path),
            this.mediaRepository.generateThumbnail(data, { ...image.preview, ...thumbnailOptions }, previewFile.path),
        ];
        let fullsizeFile;
        if (convertFullsize) {
            fullsizeFile = this.getImageFile(asset, {
                fileType: enum_1.AssetFileType.FullSize,
                format: image.fullsize.format,
                isEdited: useEdits,
                isProgressive: !!image.fullsize.progressive && image.fullsize.format !== enum_1.ImageFormat.Webp,
            });
            const fullsizeOptions = {
                format: image.fullsize.format,
                quality: image.fullsize.quality,
                progressive: image.fullsize.progressive,
                ...thumbnailOptions,
            };
            promises.push(this.mediaRepository.generateThumbnail(data, fullsizeOptions, fullsizeFile.path));
        }
        else if (generateFullsize && extracted && extracted.format === enum_1.RawExtractedFormat.Jpeg) {
            fullsizeFile = this.getImageFile(asset, {
                fileType: enum_1.AssetFileType.FullSize,
                format: extracted.format,
                isEdited: false,
                isProgressive: !!image.fullsize.progressive && image.fullsize.format !== enum_1.ImageFormat.Webp,
            });
            this.storageCore.ensureFolders(fullsizeFile.path);
            await this.storageRepository.createOrOverwriteFile(fullsizeFile.path, extracted.buffer);
            await this.mediaRepository.writeExif({
                orientation: asset.exifInfo.orientation,
                colorspace: asset.exifInfo.colorspace,
            }, fullsizeFile.path);
        }
        const outputs = await Promise.all(promises);
        if (asset.exifInfo.projectionType === 'EQUIRECTANGULAR') {
            const promises = [
                this.mediaRepository.copyTagGroup('XMP-GPano', asset.originalPath, previewFile.path),
                fullsizeFile
                    ? this.mediaRepository.copyTagGroup('XMP-GPano', asset.originalPath, fullsizeFile.path)
                    : Promise.resolve(),
            ];
            await Promise.all(promises);
        }
        const decodedDimensions = { width: info.width, height: info.height };
        const fullsizeDimensions = useEdits ? (0, transform_1.getOutputDimensions)(asset.edits, decodedDimensions) : decodedDimensions;
        return {
            files: fullsizeFile ? [previewFile, thumbnailFile, fullsizeFile] : [previewFile, thumbnailFile],
            thumbhash: outputs[0],
            fullsizeDimensions,
        };
    }
    async handleGeneratePersonThumbnail({ id }) {
        const { machineLearning, metadata, image } = await this.getConfig({ withCache: true });
        if (!(0, misc_1.isFacialRecognitionEnabled)(machineLearning) && !(0, misc_1.isFaceImportEnabled)(metadata)) {
            return enum_1.JobStatus.Skipped;
        }
        const data = await this.personRepository.getDataForThumbnailGenerationJob(id);
        if (!data) {
            this.logger.error(`Could not generate person thumbnail for ${id}: missing data`);
            return enum_1.JobStatus.Failed;
        }
        const { ownerId, x1, y1, x2, y2, oldWidth, oldHeight, exifOrientation, previewPath, originalPath } = data;
        let inputImage;
        if (data.type === enum_1.AssetType.Video) {
            if (!previewPath) {
                this.logger.error(`Could not generate person thumbnail for video ${id}: missing preview path`);
                return enum_1.JobStatus.Failed;
            }
            inputImage = previewPath;
        }
        else if (image.extractEmbedded && mime_types_1.mimeTypes.isRaw(originalPath)) {
            const extracted = await this.extractImage(originalPath, image.preview.size);
            inputImage = extracted ? extracted.buffer : originalPath;
        }
        else {
            inputImage = originalPath;
        }
        const { data: decodedImage, info } = await this.mediaRepository.decodeImage(inputImage, {
            colorspace: image.colorspace,
            processInvalidImages: process.env.IMMICH_PROCESS_INVALID_IMAGES === 'true',
            orientation: Buffer.isBuffer(inputImage) && exifOrientation ? Number(exifOrientation) : undefined,
        });
        const thumbnailPath = storage_core_1.StorageCore.getPersonThumbnailPath({ id, ownerId });
        this.storageCore.ensureFolders(thumbnailPath);
        const thumbnailOptions = {
            colorspace: image.colorspace,
            format: enum_1.ImageFormat.Jpeg,
            raw: info,
            quality: image.thumbnail.quality,
            progressive: false,
            processInvalidImages: false,
            size: constants_1.FACE_THUMBNAIL_SIZE,
            edits: [
                {
                    action: editing_dto_1.AssetEditAction.Crop,
                    parameters: this.getCrop({ old: { width: oldWidth, height: oldHeight }, new: { width: info.width, height: info.height } }, { x1, y1, x2, y2 }),
                },
            ],
        };
        await this.mediaRepository.generateThumbnail(decodedImage, thumbnailOptions, thumbnailPath);
        await this.personRepository.update({ id, thumbnailPath });
        return enum_1.JobStatus.Success;
    }
    getCrop(dims, { x1, y1, x2, y2 }) {
        const clampedX1 = (0, misc_1.clamp)(x1, 0, dims.old.width);
        const clampedY1 = (0, misc_1.clamp)(y1, 0, dims.old.height);
        const clampedX2 = (0, misc_1.clamp)(x2, 0, dims.old.width);
        const clampedY2 = (0, misc_1.clamp)(y2, 0, dims.old.height);
        const widthScale = dims.new.width / dims.old.width;
        const heightScale = dims.new.height / dims.old.height;
        const halfWidth = (widthScale * (clampedX2 - clampedX1)) / 2;
        const halfHeight = (heightScale * (clampedY2 - clampedY1)) / 2;
        const middleX = Math.round(widthScale * clampedX1 + halfWidth);
        const middleY = Math.round(heightScale * clampedY1 + halfHeight);
        const targetHalfSize = Math.floor(Math.max(halfWidth, halfHeight) * 1.1);
        const newHalfSize = Math.min(middleX - Math.max(0, middleX - targetHalfSize), middleY - Math.max(0, middleY - targetHalfSize), Math.min(dims.new.width - 1, middleX + targetHalfSize) - middleX, Math.min(dims.new.height - 1, middleY + targetHalfSize) - middleY);
        return {
            x: middleX - newHalfSize,
            y: middleY - newHalfSize,
            width: newHalfSize * 2,
            height: newHalfSize * 2,
        };
    }
    async generateVideoThumbnails(asset, { ffmpeg, image }) {
        const previewFile = this.getImageFile(asset, {
            fileType: enum_1.AssetFileType.Preview,
            format: image.preview.format,
            isEdited: false,
            isProgressive: false,
        });
        const thumbnailFile = this.getImageFile(asset, {
            fileType: enum_1.AssetFileType.Thumbnail,
            format: image.thumbnail.format,
            isEdited: false,
            isProgressive: false,
        });
        this.storageCore.ensureFolders(previewFile.path);
        const { format, audioStreams, videoStreams } = await this.mediaRepository.probe(asset.originalPath);
        const mainVideoStream = this.getMainStream(videoStreams);
        if (!mainVideoStream) {
            throw new Error(`No video streams found for asset ${asset.id}`);
        }
        const mainAudioStream = this.getMainStream(audioStreams);
        const previewConfig = media_1.ThumbnailConfig.create({ ...ffmpeg, targetResolution: image.preview.size.toString() });
        const thumbnailConfig = media_1.ThumbnailConfig.create({ ...ffmpeg, targetResolution: image.thumbnail.size.toString() });
        const previewOptions = previewConfig.getCommand(enum_1.TranscodeTarget.Video, mainVideoStream, mainAudioStream, format);
        const thumbnailOptions = thumbnailConfig.getCommand(enum_1.TranscodeTarget.Video, mainVideoStream, mainAudioStream, format);
        await this.mediaRepository.transcode(asset.originalPath, previewFile.path, previewOptions);
        await this.mediaRepository.transcode(asset.originalPath, thumbnailFile.path, thumbnailOptions);
        const thumbhash = await this.mediaRepository.generateThumbhash(previewFile.path, {
            colorspace: image.colorspace,
            processInvalidImages: process.env.IMMICH_PROCESS_INVALID_IMAGES === 'true',
        });
        return {
            files: [previewFile, thumbnailFile],
            thumbhash,
            fullsizeDimensions: { width: mainVideoStream.width, height: mainVideoStream.height },
        };
    }
    async handleQueueVideoConversion(job) {
        const { force } = job;
        let queue = [];
        for await (const asset of this.assetJobRepository.streamForVideoConversion(force)) {
            queue.push({ name: enum_1.JobName.AssetEncodeVideo, data: { id: asset.id } });
            if (queue.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await this.jobRepository.queueAll(queue);
                queue = [];
            }
        }
        await this.jobRepository.queueAll(queue);
        return enum_1.JobStatus.Success;
    }
    async handleVideoConversion({ id }) {
        const asset = await this.assetJobRepository.getForVideoConversion(id);
        if (!asset) {
            return enum_1.JobStatus.Failed;
        }
        const input = asset.originalPath;
        const output = storage_core_1.StorageCore.getEncodedVideoPath(asset);
        this.storageCore.ensureFolders(output);
        const { videoStreams, audioStreams, format } = await this.mediaRepository.probe(input, {
            countFrames: this.logger.isLevelEnabled(enum_1.LogLevel.Debug),
        });
        const videoStream = this.getMainStream(videoStreams);
        const audioStream = this.getMainStream(audioStreams);
        if (!videoStream || !format.formatName) {
            return enum_1.JobStatus.Failed;
        }
        if (!videoStream.height || !videoStream.width) {
            this.logger.warn(`Skipped transcoding for asset ${asset.id}: no video streams found`);
            return enum_1.JobStatus.Failed;
        }
        let { ffmpeg } = await this.getConfig({ withCache: true });
        const target = this.getTranscodeTarget(ffmpeg, videoStream, audioStream);
        if (target === enum_1.TranscodeTarget.None && !this.isRemuxRequired(ffmpeg, format)) {
            if (asset.encodedVideoPath) {
                this.logger.log(`Transcoded video exists for asset ${asset.id}, but is no longer required. Deleting...`);
                await this.jobRepository.queue({ name: enum_1.JobName.FileDelete, data: { files: [asset.encodedVideoPath] } });
                await this.assetRepository.update({ id: asset.id, encodedVideoPath: null });
            }
            else {
                this.logger.verbose(`Asset ${asset.id} does not require transcoding based on current policy, skipping`);
            }
            return enum_1.JobStatus.Skipped;
        }
        const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
        if (ffmpeg.accel === enum_1.TranscodeHardwareAcceleration.Disabled) {
            this.logger.log(`Transcoding video ${asset.id} without hardware acceleration`);
        }
        else {
            this.logger.log(`Transcoding video ${asset.id} with ${ffmpeg.accel.toUpperCase()}-accelerated encoding and${ffmpeg.accelDecode ? '' : ' software'} decoding`);
        }
        try {
            await this.mediaRepository.transcode(input, output, command);
        }
        catch (error) {
            this.logger.error(`Error occurred during transcoding: ${error.message}`);
            if (ffmpeg.accel === enum_1.TranscodeHardwareAcceleration.Disabled) {
                return enum_1.JobStatus.Failed;
            }
            let partialFallbackSuccess = false;
            if (ffmpeg.accelDecode) {
                try {
                    this.logger.error(`Retrying with ${ffmpeg.accel.toUpperCase()}-accelerated encoding and software decoding`);
                    ffmpeg = { ...ffmpeg, accelDecode: false };
                    const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
                    await this.mediaRepository.transcode(input, output, command);
                    partialFallbackSuccess = true;
                }
                catch (error) {
                    this.logger.error(`Error occurred during transcoding: ${error.message}`);
                }
            }
            if (!partialFallbackSuccess) {
                this.logger.error(`Retrying with ${ffmpeg.accel.toUpperCase()} acceleration disabled`);
                ffmpeg = { ...ffmpeg, accel: enum_1.TranscodeHardwareAcceleration.Disabled };
                const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
                await this.mediaRepository.transcode(input, output, command);
            }
        }
        this.logger.log(`Successfully encoded ${asset.id}`);
        await this.assetRepository.update({ id: asset.id, encodedVideoPath: output });
        return enum_1.JobStatus.Success;
    }
    getMainStream(streams) {
        return streams
            .filter((stream) => stream.codecName !== 'unknown')
            .toSorted((stream1, stream2) => stream2.bitrate - stream1.bitrate)[0];
    }
    getTranscodeTarget(config, videoStream, audioStream) {
        const isAudioTranscodeRequired = this.isAudioTranscodeRequired(config, audioStream);
        const isVideoTranscodeRequired = this.isVideoTranscodeRequired(config, videoStream);
        if (isAudioTranscodeRequired && isVideoTranscodeRequired) {
            return enum_1.TranscodeTarget.All;
        }
        if (isAudioTranscodeRequired) {
            return enum_1.TranscodeTarget.Audio;
        }
        if (isVideoTranscodeRequired) {
            return enum_1.TranscodeTarget.Video;
        }
        return enum_1.TranscodeTarget.None;
    }
    isAudioTranscodeRequired(ffmpegConfig, stream) {
        if (!stream) {
            return false;
        }
        switch (ffmpegConfig.transcode) {
            case enum_1.TranscodePolicy.Disabled: {
                return false;
            }
            case enum_1.TranscodePolicy.All: {
                return true;
            }
            case enum_1.TranscodePolicy.Required:
            case enum_1.TranscodePolicy.Optimal:
            case enum_1.TranscodePolicy.Bitrate: {
                return !ffmpegConfig.acceptedAudioCodecs.includes(stream.codecName);
            }
            default: {
                throw new Error(`Unsupported transcode policy: ${ffmpegConfig.transcode}`);
            }
        }
    }
    isVideoTranscodeRequired(ffmpegConfig, stream) {
        const scalingEnabled = ffmpegConfig.targetResolution !== 'original';
        const targetRes = Number.parseInt(ffmpegConfig.targetResolution);
        const isLargerThanTargetRes = scalingEnabled && Math.min(stream.height, stream.width) > targetRes;
        const isLargerThanTargetBitrate = stream.bitrate > this.parseBitrateToBps(ffmpegConfig.maxBitrate);
        const isTargetVideoCodec = ffmpegConfig.acceptedVideoCodecs.includes(stream.codecName);
        const isRequired = !isTargetVideoCodec || !stream.pixelFormat.endsWith('420p');
        switch (ffmpegConfig.transcode) {
            case enum_1.TranscodePolicy.Disabled: {
                return false;
            }
            case enum_1.TranscodePolicy.All: {
                return true;
            }
            case enum_1.TranscodePolicy.Required: {
                return isRequired;
            }
            case enum_1.TranscodePolicy.Optimal: {
                return isRequired || isLargerThanTargetRes;
            }
            case enum_1.TranscodePolicy.Bitrate: {
                return isRequired || isLargerThanTargetBitrate;
            }
            default: {
                throw new Error(`Unsupported transcode policy: ${ffmpegConfig.transcode}`);
            }
        }
    }
    isRemuxRequired(ffmpegConfig, { formatName, formatLongName }) {
        if (ffmpegConfig.transcode === enum_1.TranscodePolicy.Disabled) {
            return false;
        }
        const name = formatLongName === 'QuickTime / MOV' ? enum_1.VideoContainer.Mov : formatName;
        return name !== enum_1.VideoContainer.Mp4 && !ffmpegConfig.acceptedContainers.includes(name);
    }
    isSRGB({ colorspace, profileDescription, bitsPerSample }) {
        if (colorspace || profileDescription) {
            return [colorspace, profileDescription].some((s) => s?.toLowerCase().includes('srgb'));
        }
        else if (bitsPerSample) {
            return bitsPerSample === 8;
        }
        else {
            return true;
        }
    }
    parseBitrateToBps(bitrateString) {
        const bitrateValue = Number.parseInt(bitrateString);
        if (Number.isNaN(bitrateValue)) {
            return 0;
        }
        if (bitrateString.toLowerCase().endsWith('k')) {
            return bitrateValue * 1000;
        }
        else if (bitrateString.toLowerCase().endsWith('m')) {
            return bitrateValue * 1_000_000;
        }
        else {
            return bitrateValue;
        }
    }
    async shouldUseExtractedImage(extractedPathOrBuffer, targetSize) {
        const { width, height } = await this.mediaRepository.getImageDimensions(extractedPathOrBuffer);
        const extractedSize = Math.min(width, height);
        return extractedSize >= targetSize;
    }
    async getDevices() {
        try {
            return await this.storageRepository.readdir('/dev/dri');
        }
        catch {
            this.logger.debug('No devices found in /dev/dri.');
            return [];
        }
    }
    async hasMaliOpenCL() {
        try {
            const [maliIcdStat, maliDeviceStat] = await Promise.all([
                this.storageRepository.stat('/etc/OpenCL/vendors/mali.icd'),
                this.storageRepository.stat('/dev/mali0'),
            ]);
            return maliIcdStat.isFile() && maliDeviceStat.isCharacterDevice();
        }
        catch {
            this.logger.debug('OpenCL not available for transcoding, so RKMPP acceleration will use CPU tonemapping');
            return false;
        }
    }
    async syncFiles(oldFiles, newFiles) {
        const toUpsert = [];
        const pathsToDelete = [];
        const toDelete = new Set(oldFiles);
        for (const newFile of newFiles) {
            const existingFile = oldFiles.find((file) => file.type === newFile.type && file.isEdited === newFile.isEdited);
            if (existingFile) {
                toDelete.delete(existingFile);
            }
            if (existingFile?.path !== newFile.path || existingFile.isProgressive !== newFile.isProgressive) {
                toUpsert.push(newFile);
                if (existingFile && existingFile.path !== newFile.path) {
                    this.logger.debug(`Deleting old ${newFile.type} image for asset ${newFile.assetId} in favor of a replacement`);
                    pathsToDelete.push(existingFile.path);
                }
            }
        }
        if (toUpsert.length > 0) {
            await this.assetRepository.upsertFiles(toUpsert);
        }
        if (toDelete.size > 0) {
            const toDeleteArray = [...toDelete];
            for (const file of toDeleteArray) {
                pathsToDelete.push(file.path);
            }
            await this.assetRepository.deleteFiles(toDeleteArray);
        }
        if (pathsToDelete.length > 0) {
            await this.jobRepository.queue({ name: enum_1.JobName.FileDelete, data: { files: pathsToDelete } });
        }
    }
    async generateEditedThumbnails(asset, config) {
        if (asset.type !== enum_1.AssetType.Image || (asset.files.length === 0 && asset.edits.length === 0)) {
            return;
        }
        const generated = asset.edits.length > 0 ? await this.generateImageThumbnails(asset, config, true) : undefined;
        const crop = asset.edits.find((e) => e.action === editing_dto_1.AssetEditAction.Crop);
        const cropBox = crop
            ? {
                x1: crop.parameters.x,
                y1: crop.parameters.y,
                x2: crop.parameters.x + crop.parameters.width,
                y2: crop.parameters.y + crop.parameters.height,
            }
            : undefined;
        const originalDimensions = (0, asset_util_1.getDimensions)(asset.exifInfo);
        const assetFaces = await this.personRepository.getFaces(asset.id, {});
        const ocrData = await this.ocrRepository.getByAssetId(asset.id, {});
        const faceStatuses = (0, editor_1.checkFaceVisibility)(assetFaces, originalDimensions, cropBox);
        await this.personRepository.updateVisibility(faceStatuses.visible, faceStatuses.hidden);
        const ocrStatuses = (0, editor_1.checkOcrVisibility)(ocrData, originalDimensions, cropBox);
        await this.ocrRepository.updateOcrVisibilities(asset.id, ocrStatuses.visible, ocrStatuses.hidden);
        return generated;
    }
    getImageFile(asset, options) {
        const path = storage_core_1.StorageCore.getImagePath(asset, options);
        return {
            assetId: asset.id,
            type: options.fileType,
            path,
            isEdited: options.isEdited,
            isProgressive: options.isProgressive,
        };
    }
};
exports.MediaService = MediaService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AppBootstrap' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetGenerateThumbnailsQueueAll, queue: enum_1.QueueName.ThumbnailGeneration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueGenerateThumbnails", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.FileMigrationQueueAll, queue: enum_1.QueueName.Migration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueMigration", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetFileMigration, queue: enum_1.QueueName.Migration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleAssetMigration", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetEditThumbnailGeneration, queue: enum_1.QueueName.Editor }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleAssetEditThumbnailGeneration", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetGenerateThumbnails, queue: enum_1.QueueName.ThumbnailGeneration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleGenerateThumbnails", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.PersonGenerateThumbnail, queue: enum_1.QueueName.ThumbnailGeneration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleGeneratePersonThumbnail", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetEncodeVideoQueueAll, queue: enum_1.QueueName.VideoConversion }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueVideoConversion", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetEncodeVideo, queue: enum_1.QueueName.VideoConversion }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleVideoConversion", null);
exports.MediaService = MediaService = __decorate([
    (0, common_1.Injectable)()
], MediaService);
//# sourceMappingURL=media.service.js.map