"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.requireElevatedPermission = exports.checkAccess = exports.requireAccess = exports.requireUploadAccess = exports.isGranted = void 0;
const common_1 = require("@nestjs/common");
const enum_1 = require("../enum");
const set_1 = require("./set");
const isGranted = ({ requested, current }) => {
    if (current.includes(enum_1.Permission.All)) {
        return true;
    }
    return (0, set_1.setIsSuperset)(new Set(current), new Set(requested));
};
exports.isGranted = isGranted;
const requireUploadAccess = (auth) => {
    if (!auth || (auth.sharedLink && !auth.sharedLink.allowUpload)) {
        throw new common_1.UnauthorizedException();
    }
    return auth;
};
exports.requireUploadAccess = requireUploadAccess;
const requireAccess = async (access, request) => {
    const allowedIds = await (0, exports.checkAccess)(access, request);
    if (!(0, set_1.setIsEqual)(new Set(request.ids), allowedIds)) {
        throw new common_1.BadRequestException(`Not found or no ${request.permission} access`);
    }
};
exports.requireAccess = requireAccess;
const checkAccess = async (access, { ids, auth, permission }) => {
    const idSet = Array.isArray(ids) ? new Set(ids) : ids;
    if (idSet.size === 0) {
        return new Set();
    }
    return auth.sharedLink
        ? checkSharedLinkAccess(access, { sharedLink: auth.sharedLink, permission, ids: idSet })
        : checkOtherAccess(access, { auth, permission, ids: idSet });
};
exports.checkAccess = checkAccess;
const checkSharedLinkAccess = async (access, request) => {
    const { sharedLink, permission, ids } = request;
    const sharedLinkId = sharedLink.id;
    switch (permission) {
        case enum_1.Permission.AssetRead: {
            return await access.asset.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.AssetView: {
            return await access.asset.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.AssetDownload: {
            return sharedLink.allowDownload ? await access.asset.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        case enum_1.Permission.AssetUpload: {
            return sharedLink.allowUpload ? ids : new Set();
        }
        case enum_1.Permission.AssetShare: {
            return await access.asset.checkOwnerAccess(sharedLink.userId, ids, false);
        }
        case enum_1.Permission.AlbumRead: {
            return await access.album.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.AlbumDownload: {
            return sharedLink.allowDownload ? await access.album.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        case enum_1.Permission.AlbumAssetCreate: {
            return sharedLink.allowUpload ? await access.album.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        default: {
            return new Set();
        }
    }
};
const checkOtherAccess = async (access, request) => {
    const { auth, permission, ids } = request;
    switch (permission) {
        case enum_1.Permission.ActivityCreate: {
            return await access.activity.checkCreateAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ActivityDelete: {
            const isOwner = await access.activity.checkOwnerAccess(auth.user.id, ids);
            const isAlbumOwner = await access.activity.checkAlbumOwnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isAlbumOwner);
        }
        case enum_1.Permission.AssetRead: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.AssetShare: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids, false);
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isPartner);
        }
        case enum_1.Permission.AssetView: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.AssetDownload: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.AssetUpdate: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AssetDelete: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AssetCopy: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AssetEditGet: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AssetEditCreate: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AssetEditDelete: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids, auth.session?.hasElevatedPermission);
        }
        case enum_1.Permission.AlbumRead: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.Viewer);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.AlbumAssetCreate: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.Editor);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.AlbumUpdate: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.AlbumDelete: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.AlbumShare: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.AlbumDownload: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.Viewer);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.AlbumAssetDelete: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.Editor);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.AssetUpload: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.ArchiveRead: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.AuthDeviceDelete: {
            return await access.authDevice.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.FaceDelete: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.NotificationRead:
        case enum_1.Permission.NotificationUpdate:
        case enum_1.Permission.NotificationDelete: {
            return access.notification.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.TagAsset:
        case enum_1.Permission.TagRead:
        case enum_1.Permission.TagUpdate:
        case enum_1.Permission.TagDelete: {
            return await access.tag.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.TimelineRead: {
            const isOwner = ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
            const isPartner = await access.timeline.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isPartner);
        }
        case enum_1.Permission.TimelineDownload: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.MemoryRead: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.MemoryUpdate: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.MemoryDelete: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PersonCreate: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PersonRead:
        case enum_1.Permission.PersonUpdate:
        case enum_1.Permission.PersonDelete:
        case enum_1.Permission.PersonMerge: {
            return await access.person.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PersonReassign: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PartnerUpdate: {
            return await access.partner.checkUpdateAccess(auth.user.id, ids);
        }
        case enum_1.Permission.SessionRead:
        case enum_1.Permission.SessionUpdate:
        case enum_1.Permission.SessionDelete:
        case enum_1.Permission.SessionLock: {
            return access.session.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.StackRead: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.StackUpdate: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.StackDelete: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.WorkflowRead:
        case enum_1.Permission.WorkflowUpdate:
        case enum_1.Permission.WorkflowDelete: {
            return access.workflow.checkOwnerAccess(auth.user.id, ids);
        }
        default: {
            return new Set();
        }
    }
};
const requireElevatedPermission = (auth) => {
    if (!auth.session?.hasElevatedPermission) {
        throw new common_1.UnauthorizedException('Elevated permission is required');
    }
};
exports.requireElevatedPermission = requireElevatedPermission;
//# sourceMappingURL=access.js.map