"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnsupportedPostgresError = void 0;
exports.isValidDatabaseBackupName = isValidDatabaseBackupName;
exports.isValidDatabaseRoutineBackupName = isValidDatabaseRoutineBackupName;
exports.isFailedDatabaseBackupName = isFailedDatabaseBackupName;
exports.findVersion = findVersion;
exports.buildPostgresLaunchArguments = buildPostgresLaunchArguments;
exports.createDatabaseBackup = createDatabaseBackup;
exports.restoreDatabaseBackup = restoreDatabaseBackup;
exports.deleteDatabaseBackup = deleteDatabaseBackup;
exports.listDatabaseBackups = listDatabaseBackups;
exports.uploadDatabaseBackup = uploadDatabaseBackup;
exports.downloadDatabaseBackup = downloadDatabaseBackup;
const common_1 = require("@nestjs/common");
const lodash_1 = require("lodash");
const luxon_1 = require("luxon");
const node_path_1 = __importStar(require("node:path"));
const node_stream_1 = require("node:stream");
const promises_1 = require("node:stream/promises");
const semver_1 = __importDefault(require("semver"));
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const enum_1 = require("../enum");
function isValidDatabaseBackupName(filename) {
    return filename.match(/^[\d\w-.]+\.sql(?:\.gz)?$/);
}
function isValidDatabaseRoutineBackupName(filename) {
    const oldBackupStyle = filename.match(/^immich-db-backup-\d+\.sql\.gz$/);
    const newBackupStyle = filename.match(/^immich-db-backup-\d{8}T\d{6}-v.*-pg.*\.sql\.gz$/);
    return oldBackupStyle || newBackupStyle;
}
function isFailedDatabaseBackupName(filename) {
    return filename.match(/^immich-db-backup-.*\.sql\.gz\.tmp$/);
}
function findVersion(filename) {
    return /-v(.*)-/.exec(filename)?.[1];
}
class UnsupportedPostgresError extends Error {
    constructor(databaseVersion) {
        super(`Unsupported PostgreSQL version: ${databaseVersion}`);
    }
}
exports.UnsupportedPostgresError = UnsupportedPostgresError;
async function buildPostgresLaunchArguments({ logger, config, database }, bin, options = {}) {
    const { database: { config: databaseConfig }, } = config.getEnv();
    const isUrlConnection = databaseConfig.connectionType === 'url';
    const databaseVersion = await database.getPostgresVersion();
    const databaseSemver = semver_1.default.coerce(databaseVersion);
    const databaseMajorVersion = databaseSemver?.major;
    const args = [];
    if (isUrlConnection) {
        if (bin !== 'pg_dump') {
            args.push('--dbname');
        }
        let url = databaseConfig.url;
        if (URL.canParse(databaseConfig.url)) {
            const parsedUrl = new URL(databaseConfig.url);
            parsedUrl.searchParams.delete('uselibpqcompat');
            if (options.username) {
                parsedUrl.username = options.username;
            }
            url = parsedUrl.toString();
        }
        args.push(url);
    }
    else {
        args.push('--username', options.username ?? databaseConfig.username, '--host', databaseConfig.host, '--port', databaseConfig.port.toString());
        switch (bin) {
            case 'pg_dumpall': {
                args.push('--database');
                break;
            }
            case 'psql': {
                args.push('--dbname');
                break;
            }
        }
        args.push(databaseConfig.database);
    }
    switch (bin) {
        case 'pg_dump':
        case 'pg_dumpall': {
            args.push('--clean', '--if-exists');
            break;
        }
        case 'psql': {
            if (options.singleTransaction) {
                args.push('--single-transaction', '--set', 'ON_ERROR_STOP=on');
            }
            args.push('--echo-all', '--output=/dev/null');
            break;
        }
    }
    if (!databaseMajorVersion || !databaseSemver || !semver_1.default.satisfies(databaseSemver, '>=14.0.0 <19.0.0')) {
        logger.error(`Database Restore Failure: Unsupported PostgreSQL version: ${databaseVersion}`);
        throw new UnsupportedPostgresError(databaseVersion);
    }
    return {
        bin: `/usr/lib/postgresql/${databaseMajorVersion}/bin/${bin}`,
        args,
        databasePassword: isUrlConnection ? new URL(databaseConfig.url).password : databaseConfig.password,
        databaseVersion,
        databaseMajorVersion,
    };
}
async function createDatabaseBackup({ logger, storage, process: processRepository, ...pgRepos }, filenamePrefix = '') {
    logger.debug(`Database Backup Started`);
    const { bin, args, databasePassword, databaseVersion, databaseMajorVersion } = await buildPostgresLaunchArguments({ logger, ...pgRepos }, 'pg_dump');
    logger.log(`Database Backup Starting. Database Version: ${databaseMajorVersion}`);
    const filename = `${filenamePrefix}immich-db-backup-${luxon_1.DateTime.now().toFormat("yyyyLLdd'T'HHmmss")}-v${constants_1.serverVersion.toString()}-pg${databaseVersion.split(' ')[0]}.sql.gz`;
    const backupFilePath = (0, node_path_1.join)(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), filename);
    const temporaryFilePath = `${backupFilePath}.tmp`;
    try {
        const pgdump = processRepository.spawnDuplexStream(bin, args, {
            env: {
                PATH: process.env.PATH,
                PGPASSWORD: databasePassword,
            },
        });
        const gzip = processRepository.spawnDuplexStream('gzip', ['--rsyncable']);
        const fileStream = storage.createWriteStream(temporaryFilePath);
        await (0, promises_1.pipeline)(pgdump, gzip, fileStream);
        await storage.rename(temporaryFilePath, backupFilePath);
    }
    catch (error) {
        logger.error(`Database Backup Failure: ${error}`);
        await storage
            .unlink(temporaryFilePath)
            .catch((error) => logger.error(`Failed to delete failed backup file: ${error}`));
        throw error;
    }
    logger.log(`Database Backup Success`);
    return backupFilePath;
}
const SQL_DROP_CONNECTIONS = `
  -- drop all other database connections
  SELECT pg_terminate_backend(pid)
  FROM pg_stat_activity
  WHERE datname = current_database()
    AND pid <> pg_backend_pid();
`;
const SQL_RESET_SCHEMA = `
  -- re-create the default schema
  DROP SCHEMA public CASCADE;
  CREATE SCHEMA public;

  -- restore access to schema
  GRANT ALL ON SCHEMA public TO postgres;
  GRANT ALL ON SCHEMA public TO public;
`;
async function* sql(inputStream, isPgClusterDump) {
    yield SQL_DROP_CONNECTIONS;
    yield isPgClusterDump
        ? String.raw `
        \c postgres
      `
        : SQL_RESET_SCHEMA;
    for await (const chunk of inputStream) {
        yield chunk;
    }
}
async function* sqlRollback(inputStream, isPgClusterDump) {
    yield SQL_DROP_CONNECTIONS;
    if (isPgClusterDump) {
        yield String.raw `
      -- try to create database
      -- may fail but script will continue running
      CREATE DATABASE immich;

      -- switch to database / newly created database
      \c immich
    `;
    }
    yield SQL_RESET_SCHEMA;
    for await (const chunk of inputStream) {
        yield chunk;
    }
}
async function restoreDatabaseBackup({ logger, storage, process: processRepository, database: databaseRepository, health, ...pgRepos }, filename, progressCb) {
    logger.debug(`Database Restore Started`);
    let complete = false;
    try {
        if (!isValidDatabaseBackupName(filename)) {
            throw new Error('Invalid backup file format!');
        }
        const backupFilePath = node_path_1.default.join(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), filename);
        await storage.stat(backupFilePath);
        let isPgClusterDump = false;
        const version = findVersion(filename);
        if (version && semver_1.default.satisfies(version, '<= 2.4')) {
            isPgClusterDump = true;
        }
        const { bin, args, databasePassword, databaseMajorVersion } = await buildPostgresLaunchArguments({ logger, database: databaseRepository, ...pgRepos }, 'psql', {
            singleTransaction: !isPgClusterDump,
            username: isPgClusterDump ? 'postgres' : undefined,
        });
        progressCb?.('backup', 0.05);
        const restorePointFilePath = await createDatabaseBackup({ logger, storage, process: processRepository, database: databaseRepository, ...pgRepos }, 'restore-point-');
        logger.log(`Database Restore Starting. Database Version: ${databaseMajorVersion}`);
        let inputStream;
        if (backupFilePath.endsWith('.gz')) {
            const fileStream = storage.createPlainReadStream(backupFilePath);
            const gunzip = storage.createGunzip();
            fileStream.pipe(gunzip);
            inputStream = gunzip;
        }
        else {
            inputStream = storage.createPlainReadStream(backupFilePath);
        }
        const sqlStream = node_stream_1.Readable.from(sql(inputStream, isPgClusterDump));
        const psql = processRepository.spawnDuplexStream(bin, args, {
            env: {
                PATH: process.env.PATH,
                PGPASSWORD: databasePassword,
            },
        });
        const [progressSource, progressSink] = createSqlProgressStreams((progress) => {
            if (complete) {
                return;
            }
            logger.log(`Restore progress ~ ${(progress * 100).toFixed(2)}%`);
            progressCb?.('restore', progress);
        });
        await (0, promises_1.pipeline)(sqlStream, progressSource, psql, progressSink);
        try {
            progressCb?.('migrations', 0.9);
            await databaseRepository.runMigrations();
            await health.checkApiHealth();
        }
        catch (error) {
            progressCb?.('rollback', 0);
            const fileStream = storage.createPlainReadStream(restorePointFilePath);
            const gunzip = storage.createGunzip();
            fileStream.pipe(gunzip);
            inputStream = gunzip;
            const sqlStream = node_stream_1.Readable.from(sqlRollback(inputStream, isPgClusterDump));
            const psql = processRepository.spawnDuplexStream(bin, args, {
                env: {
                    PATH: process.env.PATH,
                    PGPASSWORD: databasePassword,
                },
            });
            const [progressSource, progressSink] = createSqlProgressStreams((progress) => {
                if (complete) {
                    return;
                }
                logger.log(`Rollback progress ~ ${(progress * 100).toFixed(2)}%`);
                progressCb?.('rollback', progress);
            });
            await (0, promises_1.pipeline)(sqlStream, progressSource, psql, progressSink);
            throw error;
        }
    }
    catch (error) {
        logger.error(`Database Restore Failure: ${error}`);
        throw error;
    }
    finally {
        complete = true;
    }
    logger.log(`Database Restore Success`);
}
async function deleteDatabaseBackup({ storage }, files) {
    const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
    if (files.some((filename) => !isValidDatabaseBackupName(filename))) {
        throw new common_1.BadRequestException('Invalid backup name!');
    }
    await Promise.all(files.map((filename) => storage.unlink(node_path_1.default.join(backupsFolder, filename))));
}
async function listDatabaseBackups({ storage, }) {
    const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
    const files = await storage.readdir(backupsFolder);
    const validFiles = files
        .filter((fn) => isValidDatabaseBackupName(fn))
        .toSorted((a, b) => (a.startsWith('uploaded-') === b.startsWith('uploaded-') ? a.localeCompare(b) : 1))
        .toReversed();
    const backups = await Promise.all(validFiles.map(async (filename) => {
        const stats = await storage.stat(node_path_1.default.join(backupsFolder, filename));
        return { filename, filesize: stats.size };
    }));
    return backups;
}
async function uploadDatabaseBackup({ storage }, file) {
    const backupsFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups);
    const fn = (0, node_path_1.basename)(file.originalname);
    if (!isValidDatabaseBackupName(fn)) {
        throw new common_1.BadRequestException('Invalid backup name!');
    }
    const path = (0, node_path_1.join)(backupsFolder, `uploaded-${fn}`);
    await storage.createOrOverwriteFile(path, file.buffer);
}
function downloadDatabaseBackup(fileName) {
    if (!isValidDatabaseBackupName(fileName)) {
        throw new common_1.BadRequestException('Invalid backup name!');
    }
    const path = (0, node_path_1.join)(storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Backups), fileName);
    return {
        path,
        fileName,
        cacheControl: enum_1.CacheControl.PrivateWithoutCache,
        contentType: fileName.endsWith('.gz') ? 'application/gzip' : 'application/sql',
    };
}
function createSqlProgressStreams(cb) {
    const STDIN_START_MARKER = new TextEncoder().encode('FROM stdin');
    const STDIN_END_MARKER = new TextEncoder().encode(String.raw `\.`);
    let readingStdin = false;
    let sequenceIdx = 0;
    let linesSent = 0;
    let linesProcessed = 0;
    const startedAt = +Date.now();
    const cbDebounced = (0, lodash_1.debounce)(() => {
        const progress = source.writableEnded
            ? Math.min(1, linesProcessed / linesSent)
            :
                Math.min(0.3, 0.1 + (Date.now() - startedAt) / 1e4);
        cb(progress);
    }, 100, {
        maxWait: 100,
    });
    let lastByte = -1;
    const source = new node_stream_1.PassThrough({
        transform(chunk, _encoding, callback) {
            for (const byte of chunk) {
                if (!readingStdin && byte === 10 && lastByte !== 10) {
                    linesSent += 1;
                }
                lastByte = byte;
                const sequence = readingStdin ? STDIN_END_MARKER : STDIN_START_MARKER;
                if (sequence[sequenceIdx] === byte) {
                    sequenceIdx += 1;
                    if (sequence.length === sequenceIdx) {
                        sequenceIdx = 0;
                        readingStdin = !readingStdin;
                    }
                }
                else {
                    sequenceIdx = 0;
                }
            }
            cbDebounced();
            this.push(chunk);
            callback();
        },
    });
    const sink = new node_stream_1.Writable({
        write(chunk, _encoding, callback) {
            for (const byte of chunk) {
                if (byte === 10) {
                    linesProcessed++;
                }
            }
            cbDebounced();
            callback();
        },
    });
    return [source, sink];
}
//# sourceMappingURL=database-backups.js.map