import { StrEnum, StrEnumKeys } from "./StrEnum";
/**
 * Subset of <https://exiftool.org/TagNames/ICC_Profile.html>.
 *
 * None of these tags are writable.
 */
export interface ICCProfileTags {
    /**
     * The color space of the device associated with this ICC profile.
     *
     * Common values include "RGB ", "CMYK", "Gray", "Lab ", "XYZ ", and others.
     * Note that color space names are often padded to 4 characters.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "RGB "
     */
    ColorSpaceData?: string;
    /**
     * XYZ values of the illuminant of the Profile Connection Space (PCS).
     *
     * This is typically the D50 standard illuminant (0.9642, 1.0, 0.8249) used
     * as the reference white point for ICC color transformations. The three
     * numbers represent the X, Y, and Z tristimulus values.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "0.9642 1 0.82491"
     */
    ConnectionSpaceIlluminant?: string;
    /**
     * Physical characteristics of the device (reflective vs. transmissive,
     * glossy vs. matte, positive vs. negative media, color vs. black-and-white).
     *
     * These attributes help color management systems understand the nature of
     * the device and apply appropriate color transformations.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "Reflective, Glossy, Positive, Color"
     */
    DeviceAttributes?: string;
    /**
     * Manufacturer signature code identifying the device manufacturer.
     *
     * This is typically a 4-character code registered with the ICC, or "none"
     * for generic profiles. Common manufacturers include Adobe, Apple, Canon,
     * Nikon, and others.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "none"
     */
    DeviceManufacturer?: string;
    /**
     * Human-readable description of the device manufacturer.
     *
     * This provides a full text description of the manufacturer, often
     * including URLs or additional contact information, complementing the
     * 4-character DeviceManufacturer signature code.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "IEC http://www.iec.ch"
     */
    DeviceMfgDesc?: string;
    /**
     * Device model signature code.
     *
     * This is typically a 4-character code or short identifier that specifies
     * the particular device model. Works in conjunction with DeviceManufacturer
     * for complete device identification.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "sRGB"
     */
    DeviceModel?: string;
    /**
     * Human-readable description of the device model.
     *
     * This provides a full text description of the device model, complementing
     * the shorter DeviceModel field with more detailed information such as
     * version numbers and manufacturer names.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "sRGB v1.31 (Canon)"
     */
    DeviceModelDesc?: string;
    /**
     * Absolute luminance of emissive devices in XYZ tristimulus coordinates.
     *
     * The three numbers represent X, Y, and Z values, where the Y component
     * directly corresponds to the luminance in cd/m² as defined by CIE
     * absolute colorimetry.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "76.03647 80 87.12462"
     */
    Luminance?: string;
    /**
     * Human-readable description of the ICC profile.
     *
     * This is the primary name and description of the color profile, typically
     * including the profile name, version, and sometimes the intended use case
     * or device it was created for.
     *
     * @see https://exiftool.org/TagNames/ICC_Profile.html
     *
     * ☆☆☆☆ ✔ Example: "sRGB v1.31 (Canon)"
     */
    ProfileDescription?: string;
}
export declare const ICCProfileTagNames: StrEnum<"ColorSpaceData" | "ConnectionSpaceIlluminant" | "DeviceAttributes" | "DeviceManufacturer" | "DeviceMfgDesc" | "DeviceModel" | "DeviceModelDesc" | "Luminance" | "ProfileDescription">;
export type ICCProfileTagName = StrEnumKeys<typeof ICCProfileTagNames>;
