"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetService = void 0;
const common_1 = require("@nestjs/common");
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const constants_1 = require("../constants");
const decorators_1 = require("../decorators");
const asset_response_dto_1 = require("../dtos/asset-response.dto");
const asset_dto_1 = require("../dtos/asset.dto");
const editing_dto_1 = require("../dtos/editing.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const access_1 = require("../utils/access");
const asset_util_1 = require("../utils/asset.util");
const database_1 = require("../utils/database");
const transform_1 = require("../utils/transform");
let AssetService = class AssetService extends base_service_1.BaseService {
    async getStatistics(auth, dto) {
        if (dto.visibility === enum_1.AssetVisibility.Locked) {
            (0, access_1.requireElevatedPermission)(auth);
        }
        const stats = await this.assetRepository.getStatistics(auth.user.id, dto);
        return (0, asset_dto_1.mapStats)(stats);
    }
    async getRandom(auth, count) {
        const partnerIds = await (0, asset_util_1.getMyPartnerIds)({
            userId: auth.user.id,
            repository: this.partnerRepository,
            timelineEnabled: true,
        });
        const assets = await this.assetRepository.getRandom([auth.user.id, ...partnerIds], count);
        return assets.map((a) => (0, asset_response_dto_1.mapAsset)(a, { auth }));
    }
    async getUserAssetsByDeviceId(auth, deviceId) {
        return this.assetRepository.getAllByDeviceId(auth.user.id, deviceId);
    }
    async get(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetRead, ids: [id] });
        const asset = await this.assetRepository.getById(id, {
            exifInfo: true,
            owner: true,
            faces: { person: true },
            stack: { assets: true },
            edits: true,
            tags: true,
        });
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        if (auth.sharedLink && !auth.sharedLink.showExif) {
            return (0, asset_response_dto_1.mapAsset)(asset, { stripMetadata: true, withStack: true, auth });
        }
        const data = (0, asset_response_dto_1.mapAsset)(asset, { withStack: true, auth });
        if (auth.sharedLink) {
            delete data.owner;
        }
        if (data.ownerId !== auth.user.id || auth.sharedLink) {
            data.people = [];
        }
        return data;
    }
    async update(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: [id] });
        const { description, dateTimeOriginal, latitude, longitude, rating, ...rest } = dto;
        const repos = { asset: this.assetRepository, event: this.eventRepository };
        let previousMotion = null;
        if (rest.livePhotoVideoId) {
            await (0, asset_util_1.onBeforeLink)(repos, { userId: auth.user.id, livePhotoVideoId: rest.livePhotoVideoId });
        }
        else if (rest.livePhotoVideoId === null) {
            const asset = await this.findOrFail(id);
            if (asset.livePhotoVideoId) {
                previousMotion = await (0, asset_util_1.onBeforeUnlink)(repos, { livePhotoVideoId: asset.livePhotoVideoId });
            }
        }
        await this.updateExif({ id, description, dateTimeOriginal, latitude, longitude, rating });
        const asset = await this.assetRepository.update({ id, ...rest });
        if (previousMotion && asset) {
            await (0, asset_util_1.onAfterUnlink)(repos, {
                userId: auth.user.id,
                livePhotoVideoId: previousMotion.id,
                visibility: asset.visibility,
            });
        }
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        return (0, asset_response_dto_1.mapAsset)(asset, { auth });
    }
    async updateAll(auth, dto) {
        const { ids, isFavorite, visibility, dateTimeOriginal, latitude, longitude, rating, description, duplicateId, dateTimeRelative, timeZone, } = dto;
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids });
        const assetDto = lodash_1.default.omitBy({ isFavorite, visibility, duplicateId }, lodash_1.default.isUndefined);
        const exifDto = lodash_1.default.omitBy({
            latitude,
            longitude,
            rating,
            description,
            dateTimeOriginal,
        }, lodash_1.default.isUndefined);
        const extractedTimeZone = dateTimeOriginal ? luxon_1.DateTime.fromISO(dateTimeOriginal, { setZone: true }).zone : undefined;
        if (Object.keys(exifDto).length > 0) {
            await this.assetRepository.updateAllExif(ids, exifDto);
        }
        if ((dateTimeRelative !== undefined && dateTimeRelative !== 0) ||
            timeZone !== undefined ||
            extractedTimeZone?.type === 'fixed') {
            await this.assetRepository.updateDateTimeOriginal(ids, dateTimeRelative, timeZone ?? extractedTimeZone?.name);
        }
        if (Object.keys(assetDto).length > 0) {
            await this.assetRepository.updateAll(ids, assetDto);
        }
        if (visibility === enum_1.AssetVisibility.Locked) {
            await this.albumRepository.removeAssetsFromAll(ids);
        }
        await this.jobRepository.queueAll(ids.map((id) => ({ name: enum_1.JobName.SidecarWrite, data: { id } })));
    }
    async copy(auth, { sourceId, targetId, albums = true, sidecar = true, sharedLinks = true, stack = true, favorite = true, }) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetCopy, ids: [sourceId, targetId] });
        const sourceAsset = await this.assetRepository.getForCopy(sourceId);
        const targetAsset = await this.assetRepository.getForCopy(targetId);
        if (!sourceAsset || !targetAsset) {
            throw new common_1.BadRequestException('Both assets must exist');
        }
        if (sourceId === targetId) {
            throw new common_1.BadRequestException('Source and target id must be distinct');
        }
        if (albums) {
            await this.albumRepository.copyAlbums({ sourceAssetId: sourceId, targetAssetId: targetId });
        }
        if (sharedLinks) {
            await this.sharedLinkAssetRepository.copySharedLinks({ sourceAssetId: sourceId, targetAssetId: targetId });
        }
        if (stack) {
            await this.copyStack({ sourceAsset, targetAsset });
        }
        if (favorite) {
            await this.assetRepository.update({ id: targetId, isFavorite: sourceAsset.isFavorite });
        }
        if (sidecar) {
            await this.copySidecar({ sourceAsset, targetAsset });
        }
    }
    async copyStack({ sourceAsset, targetAsset, }) {
        if (!sourceAsset.stackId) {
            return;
        }
        if (targetAsset.stackId) {
            await this.stackRepository.merge({ sourceId: sourceAsset.stackId, targetId: targetAsset.stackId });
            await this.stackRepository.delete(sourceAsset.stackId);
        }
        else {
            await this.assetRepository.update({ id: targetAsset.id, stackId: sourceAsset.stackId });
        }
    }
    async copySidecar({ sourceAsset, targetAsset, }) {
        const { sidecarFile: sourceFile } = (0, asset_util_1.getAssetFiles)(sourceAsset.files);
        if (!sourceFile?.path) {
            return;
        }
        const { sidecarFile: targetFile } = (0, asset_util_1.getAssetFiles)(targetAsset.files ?? []);
        if (targetFile?.path) {
            await this.storageRepository.unlink(targetFile.path);
        }
        await this.storageRepository.copyFile(sourceFile.path, `${targetAsset.originalPath}.xmp`);
        await this.assetRepository.upsertFile({
            assetId: targetAsset.id,
            path: `${targetAsset.originalPath}.xmp`,
            type: enum_1.AssetFileType.Sidecar,
        });
        await this.jobRepository.queue({ name: enum_1.JobName.AssetExtractMetadata, data: { id: targetAsset.id } });
    }
    async handleAssetDeletionCheck() {
        const config = await this.getConfig({ withCache: false });
        const trashedDays = config.trash.enabled ? config.trash.days : 0;
        const trashedBefore = luxon_1.DateTime.now()
            .minus(luxon_1.Duration.fromObject({ days: trashedDays }))
            .toJSDate();
        let chunk = [];
        const queueChunk = async () => {
            if (chunk.length > 0) {
                await this.jobRepository.queueAll(chunk.map(({ id, isOffline }) => ({
                    name: enum_1.JobName.AssetDelete,
                    data: { id, deleteOnDisk: !isOffline },
                })));
                chunk = [];
            }
        };
        const assets = this.assetJobRepository.streamForDeletedJob(trashedBefore);
        for await (const asset of assets) {
            chunk.push(asset);
            if (chunk.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await queueChunk();
            }
        }
        await queueChunk();
        return enum_1.JobStatus.Success;
    }
    async handleAssetDeletion(job) {
        const { id, deleteOnDisk } = job;
        const asset = await this.assetJobRepository.getForAssetDeletion(id);
        if (!asset) {
            return enum_1.JobStatus.Failed;
        }
        if (asset.stack?.primaryAssetId === id) {
            const stackAssetIds = asset.stack?.assets.map((a) => a.id) ?? [];
            if (stackAssetIds.length > 2) {
                const newPrimaryAssetId = stackAssetIds.find((a) => a !== id);
                await this.stackRepository.update(asset.stack.id, {
                    id: asset.stack.id,
                    primaryAssetId: newPrimaryAssetId,
                });
            }
            else {
                await this.stackRepository.delete(asset.stack.id);
            }
        }
        await this.assetRepository.remove(asset);
        if (!asset.libraryId) {
            await this.userRepository.updateUsage(asset.ownerId, -(asset.exifInfo?.fileSizeInByte || 0));
        }
        await this.eventRepository.emit('AssetDelete', { assetId: id, userId: asset.ownerId });
        if (asset.livePhotoVideoId) {
            const count = await this.assetRepository.getLivePhotoCount(asset.livePhotoVideoId);
            if (count === 0) {
                await this.jobRepository.queue({
                    name: enum_1.JobName.AssetDelete,
                    data: { id: asset.livePhotoVideoId, deleteOnDisk },
                });
            }
        }
        const assetFiles = (0, asset_util_1.getAssetFiles)(asset.files ?? []);
        const files = [
            assetFiles.thumbnailFile?.path,
            assetFiles.previewFile?.path,
            assetFiles.fullsizeFile?.path,
            assetFiles.editedFullsizeFile?.path,
            assetFiles.editedPreviewFile?.path,
            assetFiles.editedThumbnailFile?.path,
            asset.encodedVideoPath,
        ];
        if (deleteOnDisk && !asset.isOffline) {
            files.push(assetFiles.sidecarFile?.path, asset.originalPath);
        }
        await this.jobRepository.queue({ name: enum_1.JobName.FileDelete, data: { files: files.filter(Boolean) } });
        return enum_1.JobStatus.Success;
    }
    async deleteAll(auth, dto) {
        const { ids, force } = dto;
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetDelete, ids });
        await this.assetRepository.updateAll(ids, {
            deletedAt: new Date(),
            status: force ? enum_1.AssetStatus.Deleted : enum_1.AssetStatus.Trashed,
        });
        await this.eventRepository.emit(force ? 'AssetDeleteAll' : 'AssetTrashAll', {
            assetIds: ids,
            userId: auth.user.id,
        });
    }
    async getMetadata(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetRead, ids: [id] });
        return this.assetRepository.getMetadata(id);
    }
    async getOcr(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetRead, ids: [id] });
        const ocr = await this.ocrRepository.getByAssetId(id);
        const asset = await this.assetRepository.getById(id, { exifInfo: true, edits: true });
        if (!asset || !asset.exifInfo || !asset.edits) {
            throw new common_1.BadRequestException('Asset not found');
        }
        const dimensions = (0, asset_util_1.getDimensions)(asset.exifInfo);
        return ocr.map((item) => (0, transform_1.transformOcrBoundingBox)(item, asset.edits, dimensions));
    }
    async upsertBulkMetadata(auth, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: dto.items.map((item) => item.assetId) });
        const uniqueKeys = new Set();
        for (const item of dto.items) {
            const key = `(${item.assetId}, ${item.key})`;
            if (uniqueKeys.has(key)) {
                throw new common_1.BadRequestException(`Duplicate items are not allowed: "${key}"`);
            }
            uniqueKeys.add(key);
        }
        return this.assetRepository.upsertBulkMetadata(dto.items);
    }
    async upsertMetadata(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: [id] });
        const uniqueKeys = new Set();
        for (const { key } of dto.items) {
            if (uniqueKeys.has(key)) {
                throw new common_1.BadRequestException(`Duplicate items are not allowed: "${key}"`);
            }
            uniqueKeys.add(key);
        }
        return this.assetRepository.upsertMetadata(id, dto.items);
    }
    async getMetadataByKey(auth, id, key) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetRead, ids: [id] });
        const item = await this.assetRepository.getMetadataByKey(id, key);
        if (!item) {
            throw new common_1.BadRequestException(`Metadata with key "${key}" not found for asset with id "${id}"`);
        }
        return item;
    }
    async deleteMetadataByKey(auth, id, key) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: [id] });
        return this.assetRepository.deleteMetadataByKey(id, key);
    }
    async deleteBulkMetadata(auth, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: dto.items.map((item) => item.assetId) });
        await this.assetRepository.deleteBulkMetadata(dto.items);
    }
    async run(auth, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetUpdate, ids: dto.assetIds });
        const jobs = [];
        for (const id of dto.assetIds) {
            switch (dto.name) {
                case asset_dto_1.AssetJobName.REFRESH_FACES: {
                    jobs.push({ name: enum_1.JobName.AssetDetectFaces, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.REFRESH_METADATA: {
                    jobs.push({ name: enum_1.JobName.AssetExtractMetadata, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.REGENERATE_THUMBNAIL: {
                    jobs.push({ name: enum_1.JobName.AssetGenerateThumbnails, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.TRANSCODE_VIDEO: {
                    jobs.push({ name: enum_1.JobName.AssetEncodeVideo, data: { id } });
                    break;
                }
            }
        }
        await this.jobRepository.queueAll(jobs);
    }
    async findOrFail(id) {
        const asset = await this.assetRepository.getById(id);
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        return asset;
    }
    async updateExif(dto) {
        const { id, description, dateTimeOriginal, latitude, longitude, rating } = dto;
        const extractedTimeZone = dateTimeOriginal ? luxon_1.DateTime.fromISO(dateTimeOriginal, { setZone: true }).zone : undefined;
        const writes = lodash_1.default.omitBy({
            description,
            dateTimeOriginal,
            timeZone: extractedTimeZone?.type === 'fixed' ? extractedTimeZone.name : undefined,
            latitude,
            longitude,
            rating,
        }, lodash_1.default.isUndefined);
        if (Object.keys(writes).length > 0) {
            await this.assetRepository.upsertExif((0, database_1.updateLockedColumns)({
                assetId: id,
                ...writes,
            }), { lockedPropertiesBehavior: 'append' });
            await this.jobRepository.queue({ name: enum_1.JobName.SidecarWrite, data: { id } });
        }
    }
    async getAssetEdits(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetRead, ids: [id] });
        const edits = await this.assetEditRepository.getAll(id);
        return {
            assetId: id,
            edits,
        };
    }
    async editAsset(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetEditCreate, ids: [id] });
        const asset = await this.assetRepository.getById(id, { exifInfo: true });
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        if (asset.type !== enum_1.AssetType.Image) {
            throw new common_1.BadRequestException('Only images can be edited');
        }
        if (asset.livePhotoVideoId) {
            throw new common_1.BadRequestException('Editing live photos is not supported');
        }
        if ((0, asset_util_1.isPanorama)(asset)) {
            throw new common_1.BadRequestException('Editing panorama images is not supported');
        }
        if (asset.originalPath?.toLowerCase().endsWith('.gif')) {
            throw new common_1.BadRequestException('Editing GIF images is not supported');
        }
        if (asset.originalPath?.toLowerCase().endsWith('.svg')) {
            throw new common_1.BadRequestException('Editing SVG images is not supported');
        }
        const cropIndex = dto.edits.findIndex((e) => e.action === editing_dto_1.AssetEditAction.Crop);
        if (cropIndex > 0) {
            throw new common_1.BadRequestException('Crop action must be the first edit action');
        }
        const crop = cropIndex === -1 ? null : dto.edits[cropIndex];
        if (crop) {
            const { width: assetWidth, height: assetHeight } = (0, asset_util_1.getDimensions)(asset.exifInfo);
            if (!assetWidth || !assetHeight) {
                throw new common_1.BadRequestException('Asset dimensions are not available for editing');
            }
            const { x, y, width, height } = crop.parameters;
            if (x + width > assetWidth || y + height > assetHeight) {
                throw new common_1.BadRequestException('Crop parameters are out of bounds');
            }
        }
        const newEdits = await this.assetEditRepository.replaceAll(id, dto.edits);
        await this.jobRepository.queue({ name: enum_1.JobName.AssetEditThumbnailGeneration, data: { id } });
        return {
            assetId: id,
            edits: newEdits,
        };
    }
    async removeAssetEdits(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AssetEditDelete, ids: [id] });
        const asset = await this.assetRepository.getById(id);
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        await this.assetEditRepository.replaceAll(id, []);
        await this.jobRepository.queue({ name: enum_1.JobName.AssetEditThumbnailGeneration, data: { id } });
    }
};
exports.AssetService = AssetService;
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetDeleteCheck, queue: enum_1.QueueName.BackgroundTask }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AssetService.prototype, "handleAssetDeletionCheck", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.AssetDelete, queue: enum_1.QueueName.BackgroundTask }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], AssetService.prototype, "handleAssetDeletion", null);
exports.AssetService = AssetService = __decorate([
    (0, common_1.Injectable)()
], AssetService);
//# sourceMappingURL=asset.service.js.map