"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compareColumns = void 0;
const helpers_1 = require("../helpers");
const types_1 = require("../types");
exports.compareColumns = {
    getRenameKey: (column) => {
        return (0, helpers_1.asRenameKey)([
            column.tableName,
            column.type,
            column.nullable,
            column.default,
            column.storage,
            column.primary,
            column.isArray,
            column.length,
            column.identity,
            column.enumName,
            column.numericPrecision,
            column.numericScale,
        ]);
    },
    onRename: (source, target) => [
        {
            type: 'ColumnRename',
            tableName: source.tableName,
            oldName: target.name,
            newName: source.name,
            reason: types_1.Reason.Rename,
        },
    ],
    onMissing: (source) => [
        {
            type: 'ColumnAdd',
            column: source,
            reason: types_1.Reason.MissingInTarget,
        },
    ],
    onExtra: (target) => [
        {
            type: 'ColumnDrop',
            tableName: target.tableName,
            columnName: target.name,
            reason: types_1.Reason.MissingInSource,
        },
    ],
    onCompare: (source, target) => {
        const sourceType = (0, helpers_1.getColumnType)(source);
        const targetType = (0, helpers_1.getColumnType)(target);
        const isTypeChanged = sourceType !== targetType;
        if (isTypeChanged) {
            return dropAndRecreateColumn(source, target, `column type is different (${sourceType} vs ${targetType})`);
        }
        const items = [];
        if (source.nullable !== target.nullable) {
            items.push({
                type: 'ColumnAlter',
                tableName: source.tableName,
                columnName: source.name,
                changes: {
                    nullable: source.nullable,
                },
                reason: `nullable is different (${source.nullable} vs ${target.nullable})`,
            });
        }
        if (!(0, helpers_1.isDefaultEqual)(source, target)) {
            items.push({
                type: 'ColumnAlter',
                tableName: source.tableName,
                columnName: source.name,
                changes: {
                    default: String(source.default ?? 'NULL'),
                },
                reason: `default is different (${source.default ?? 'null'} vs ${target.default})`,
            });
        }
        if (source.comment !== target.comment) {
            items.push({
                type: 'ColumnAlter',
                tableName: source.tableName,
                columnName: source.name,
                changes: {
                    comment: String(source.comment),
                },
                reason: `comment is different (${source.comment} vs ${target.comment})`,
            });
        }
        return items;
    },
};
const dropAndRecreateColumn = (source, target, reason) => {
    return [
        {
            type: 'ColumnDrop',
            tableName: target.tableName,
            columnName: target.name,
            reason,
        },
        { type: 'ColumnAdd', column: source, reason },
    ];
};
//# sourceMappingURL=column.comparer.js.map