"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isNumber = isNumber;
exports.isInteger = isInteger;
exports.isFloat = isFloat;
exports.toFloat = toFloat;
exports.toInt = toInt;
exports.roundToDecimalPlaces = roundToDecimalPlaces;
function isNumber(n) {
    return typeof n === "number" && isFinite(n);
}
function isInteger(n) {
    return isNumber(n) && Number.isInteger(n);
}
function isFloat(n) {
    return isNumber(n) && !Number.isInteger(n);
}
function toFloat(n) {
    if (n == null)
        return;
    if (isNumber(n))
        return n;
    try {
        const f = parseFloat(String(n).trim());
        return isNumber(f) ? f : undefined;
    }
    catch {
        return undefined;
    }
}
function toInt(n) {
    if (n == null)
        return;
    if (isNumber(n)) {
        // we don't round here, because parsing floats also doesn't round.
        return Math.floor(n);
    }
    try {
        return parseInt(String(n).trim());
    }
    catch {
        return undefined;
    }
}
/**
 * Rounds a number to a specified number of decimal places.
 *
 * @param value - The number to round
 * @param precision - The number of decimal places to retain
 * @returns The rounded number with specified precision
 *
 * @example
 * roundToDecimalPlaces(3.14159, 2) // Returns 3.14
 * roundToDecimalPlaces(123.456789, 3) // Returns 123.457
 * roundToDecimalPlaces(0.0001234, 4) // Returns 0.0001
 */
function roundToDecimalPlaces(value, precision) {
    // Handle edge cases
    if (!isNumber(value))
        throw new Error("Value must be a number");
    if (precision < 0)
        throw new Error("Precision must be non-negative");
    if (value === 0)
        return 0;
    const multiplier = Math.pow(10, precision);
    return Math.abs(value) < Number.EPSILON
        ? 0
        : Math.round(value * multiplier) / multiplier;
}
//# sourceMappingURL=Number.js.map