"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExifTime = void 0;
const luxon_1 = require("luxon");
const DateTime_1 = require("./DateTime");
const String_1 = require("./String");
const TimeParsing_1 = require("./TimeParsing");
const Timezones_1 = require("./Timezones");
/**
 * Encodes an ExifTime (which may not have a timezone offset)
 */
class ExifTime {
    hour;
    minute;
    second;
    millisecond;
    rawValue;
    inferredZone;
    static fromEXIF(text, defaultZone) {
        const s = (0, String_1.toS)(text).trim();
        if (s.length === 0)
            return;
        const result = (0, TimeParsing_1.parseDateTime)(text, (0, TimeParsing_1.timeFormats)({ defaultZone }));
        if (result != null) {
            return this.fromDateTime(result.dt, text, result.unsetZone ? undefined : (0, Timezones_1.getZoneName)({ zone: result.dt.zone }), result.inferredZone, result.unsetMilliseconds);
        }
        return;
    }
    static fromDateTime(dt, rawValue, zone, inferredZone, unsetMilliseconds) {
        return !(0, DateTime_1.validDateTime)(dt)
            ? undefined
            : new ExifTime(dt.hour, dt.minute, dt.second, unsetMilliseconds ? undefined : dt.millisecond, rawValue, zone, inferredZone);
    }
    #dt;
    #z;
    zone;
    constructor(hour, minute, second, millisecond, rawValue, zoneName, inferredZone) {
        this.hour = hour;
        this.minute = minute;
        this.second = second;
        this.millisecond = millisecond;
        this.rawValue = rawValue;
        this.inferredZone = inferredZone;
        this.zone = (0, Timezones_1.getZoneName)({ zoneName });
    }
    toDateTime() {
        return (this.#dt ??= luxon_1.DateTime.fromObject({
            hour: this.hour,
            minute: this.minute,
            second: this.second,
            millisecond: this.millisecond,
        }, {
            zone: this.zone,
        }));
    }
    /**
     * Alias for `.millisecond`
     */
    get millis() {
        return this.millisecond;
    }
    get hasZone() {
        return this.zone != null;
    }
    #subsec() {
        return this.millisecond == null ? "" : "." + (0, String_1.pad3)(this.millisecond);
    }
    #shortZone() {
        return (this.#z ??= (0, Timezones_1.zoneToShortOffset)(this.zone));
    }
    toString() {
        return ((0, String_1.pad2)(this.hour, this.minute, this.second).join(":") +
            this.#subsec() +
            this.#shortZone());
    }
    toISOString() {
        return this.toString();
    }
    toExifString() {
        return this.toString();
    }
    setZone(zone, opts) {
        const dt = (0, TimeParsing_1.setZone)({
            zone,
            src: this.toDateTime(),
            srcHasZone: this.hasZone,
            opts,
        });
        return ExifTime.fromDateTime(dt, this.rawValue, this.zone, this.inferredZone, this.millisecond == null);
    }
    toJSON() {
        return {
            _ctor: "ExifTime",
            hour: this.hour,
            minute: this.minute,
            second: this.second,
            millisecond: this.millisecond,
            rawValue: this.rawValue,
            zone: this.zone,
            inferredZone: this.inferredZone,
        };
    }
    static fromJSON(json) {
        return new ExifTime(json.hour, json.minute, json.second, json.millisecond, json.rawValue, json.zone, json.inferredZone);
    }
}
exports.ExifTime = ExifTime;
//# sourceMappingURL=ExifTime.js.map