import { StrEnum, StrEnumKeys } from "./StrEnum";
/**
 * Root structure for MWG hierarchical keyword metadata.
 *
 * Contains the top-level hierarchy of keyword structures organized in a tree format.
 */
export interface KeywordInfoStruct {
    /**
     * Array of keyword structures representing the top level of the keyword hierarchy.
     */
    Hierarchy?: KeywordStruct[];
}
/**
 * Individual keyword node in a hierarchical keyword tree.
 *
 * The MWG standard supports unlimited nesting depth, but ExifTool limits
 * extraction to 6 levels to prevent infinite recursion.
 * Example: "Nature > Animals > Mammals > Dogs".
 */
export interface KeywordStruct {
    /**
     * Array of nested keyword structures representing child keywords in the hierarchy.
     *
     * Enables multi-level keyword organization by recursively containing additional KeywordStruct entries.
     */
    Children?: KeywordStruct[];
    /**
     * The keyword text value for this node in the hierarchy.
     *
     * This is the actual keyword string that appears at this level.
     */
    Keyword: string;
    /**
     * Indicates whether this keyword has been applied/activated in the image metadata.
     */
    Applied?: boolean;
}
/**
 * MWG (Metadata Working Group) Keywords tags for standardized hierarchical keyword metadata.
 *
 * These tags enable consistent keyword organization across different metadata formats (XMP, IPTC, EXIF).
 * ExifTool extracts hierarchical keywords up to 6 levels deep to avoid infinite recursion.
 *
 * Note: this is a partial interface: ExifTool extracts more fields from this
 * group type (but I haven't seen those in the wild).
 *
 * @see https://exiftool.org/TagNames/MWG.html#Keywords
 */
export interface MWGKeywordTags {
    /**
     * Array of structured hierarchical keyword information.
     *
     * Each entry contains a Hierarchy field with nested KeywordStruct objects representing
     * the complete keyword tree. This is the primary field for accessing full keyword hierarchies.
     */
    KeywordInfo?: KeywordInfoStruct[];
    /**
     * Direct array of hierarchical keyword structures.
     *
     * Alternative representation providing direct access to keyword tree nodes without
     * the KeywordInfo wrapper. Supports multi-level keyword organization with recursive Children fields.
     */
    HierarchicalKeywords?: KeywordStruct[];
}
export declare const MWGKeywordTagNames: StrEnum<"KeywordInfo" | "HierarchicalKeywords">;
export type MWGKeywordTagName = StrEnumKeys<typeof MWGKeywordTagNames>;
/**
 * Information about a single collection that groups related images.
 *
 * Collections enable organizing images into named groups (e.g., "Summer Vacation 2024", "Portfolio Selections").
 * The MWG standard uses both a human-readable name and a unique identifier for each collection.
 */
export interface CollectionInfo {
    /**
     * Human-readable display name for the collection.
     *
     * This is the name users see when browsing collections (e.g., "Best Photos 2024").
     */
    CollectionName: string;
    /**
     * Unique identifier (URI) for the collection.
     *
     * Provides a globally unique reference to distinguish between collections with the same name.
     * Typically formatted as a URI or UUID to ensure uniqueness across different systems.
     */
    CollectionURI: string;
}
/**
 * MWG (Metadata Working Group) Collections tags for standardized image grouping metadata.
 *
 * Collections allow images to be organized into named groups while maintaining their physical location.
 * This provides a standardized way to group related images across different metadata formats (XMP, IPTC, EXIF).
 *
 * @see https://exiftool.org/TagNames/MWG.html#Collections
 */
export interface MWGCollectionsTags {
    /**
     * Array of collections that include this image.
     *
     * An image can belong to multiple collections. Each collection is identified by both a
     * human-readable name and a unique URI, enabling consistent collection management across
     * different applications and systems.
     */
    Collections?: CollectionInfo[];
}
export declare const MWGCollectionsTagNames: StrEnum<"Collections">;
export type MWGCollectionsTagName = StrEnumKeys<typeof MWGCollectionsTagNames>;
