// src/dialect.mts
import {
  PostgresAdapter,
  PostgresIntrospector,
  PostgresQueryCompiler
} from "kysely";

// src/driver.mts
import {
  PostgresDriver
} from "kysely";

// src/utils.mts
function freeze(obj) {
  return Object.freeze(obj);
}

// src/driver.mts
var RELEASE_CONNECTION_SYMBOL = Symbol("release");
var PostgresJSDriver = class extends PostgresDriver {
  #config;
  #postgres;
  constructor(config) {
    super({});
    this.#config = freeze({ ...config });
  }
  async acquireConnection() {
    const reservedConnection = await this.#postgres.reserve();
    const connection = new PostgresJSConnection(reservedConnection);
    await this.#config.onReserveConnection?.(connection);
    return connection;
  }
  async destroy() {
    await this.#postgres.end();
  }
  async init() {
    const { postgres } = this.#config;
    this.#postgres = isPostgresJSSql(postgres) ? postgres : await postgres();
  }
  async releaseConnection(connection) {
    ;
    connection[RELEASE_CONNECTION_SYMBOL]();
  }
};
function isPostgresJSSql(thing) {
  return typeof thing === "function" && "reserve" in thing;
}
var PostgresJSConnection = class {
  #reservedConnection;
  constructor(reservedConnection) {
    this.#reservedConnection = reservedConnection;
  }
  async executeQuery(compiledQuery) {
    const result = await this.#reservedConnection.unsafe(compiledQuery.sql, [
      ...compiledQuery.parameters
    ]);
    const { command, count } = result;
    return {
      numAffectedRows: command === "INSERT" || command === "UPDATE" || command === "DELETE" || command === "MERGE" ? BigInt(count) : void 0,
      rows: Array.from(result.values())
    };
  }
  async *streamQuery(compiledQuery, chunkSize) {
    if (!Number.isInteger(chunkSize) || chunkSize <= 0) {
      throw new PostgresJSDialectError("chunkSize must be a positive integer");
    }
    const query = this.#reservedConnection.unsafe(compiledQuery.sql, [
      ...compiledQuery.parameters
    ]);
    if (typeof query.cursor !== "function") {
      throw new Error(
        "PostgresJSDialect detected the instance you passed to it does not support streaming."
      );
    }
    const cursor = query.cursor(chunkSize);
    for await (const rows of cursor) {
      yield { rows };
    }
  }
  [RELEASE_CONNECTION_SYMBOL]() {
    this.#reservedConnection.release();
  }
};
var PostgresJSDialectError = class extends Error {
  constructor(message) {
    super(message);
    this.name = "PostgresJSDialectError";
  }
};

// src/dialect.mts
var PostgresJSDialect = class {
  #config;
  constructor(config) {
    this.#config = freeze({ ...config });
  }
  createAdapter() {
    return new PostgresAdapter();
  }
  createDriver() {
    return new PostgresJSDriver(this.#config);
  }
  // biome-ignore lint/suspicious/noExplicitAny: this is fine.
  createIntrospector(db) {
    return new PostgresIntrospector(db);
  }
  createQueryCompiler() {
    return new PostgresQueryCompiler();
  }
};
export {
  PostgresJSDialect,
  PostgresJSDialectError,
  PostgresJSDriver
};
