"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyProvidersResolutionTimeoutException = exports.ProxyProviderNotResolvedException = exports.ProxyProviderNotRegisteredException = exports.ProxyProviderNotDecoratedException = exports.UnknownProxyDependenciesException = exports.ProxyProviderError = void 0;
const proxy_provider_constants_1 = require("./proxy-provider.constants");
const proxy_provider_utils_1 = require("./proxy-provider.utils");
class ProxyProviderError extends Error {
    constructor() {
        super(...arguments);
        this.name = this.constructor.name;
    }
}
exports.ProxyProviderError = ProxyProviderError;
class UnknownProxyDependenciesException extends ProxyProviderError {
    static create(error, Provider) {
        const expectedParams = (0, proxy_provider_utils_1.reflectClassConstructorParams)(Provider);
        const foundParams = this.extractDependencyParams(error);
        const notFoundParamIndex = foundParams.findIndex((it) => it == '?');
        let notFoundParamName = expectedParams[notFoundParamIndex]?.name;
        const message = this.composeMessage(Provider.name, foundParams.join(', '), notFoundParamName ?? 'Unknown', notFoundParamIndex);
        return new this(message);
    }
    static extractDependencyParams(error) {
        // matches the parameters from NestJS's error message:
        // e.g: "Nest can't resolve dependencies of the Something (Cats, ?). [...]"
        // returns ['Cats', '?']
        return error.message.match(/\w+ \((.*?)\)./)?.[1].split(', ') ?? [];
    }
    static composeMessage(providerName, foundParamsString, notFoundParamName, notFoundIndex) {
        return (`Cannot create Proxy provider ${providerName} (${foundParamsString}). ` +
            `The argument ${notFoundParamName} at index [${notFoundIndex}] was not found in the ClsModule Context.\n\n` +
            `Potential solutions:\n` +
            `- If ${notFoundParamName} is a provider from a separate module, make sure to import the module in "ClsModule.forFeatureAsync()" registration`);
    }
}
exports.UnknownProxyDependenciesException = UnknownProxyDependenciesException;
class ProxyProviderNotDecoratedException extends ProxyProviderError {
    static create(Provider) {
        const message = `Cannot create a Proxy provider for ${Provider.name}. The class must be explicitly decorated with the @InjectableProxy() decorator to distinguish it from a regular provider.`;
        return new this(message);
    }
}
exports.ProxyProviderNotDecoratedException = ProxyProviderNotDecoratedException;
class ProxyProviderNotRegisteredException extends ProxyProviderError {
    static create(providerSymbol) {
        const message = `Cannot resolve a Proxy provider for symbol "${providerSymbol.description}", because it was not registered using "ClsModule.forFeature()" or "ClsModule.forFeatureAsync()".`;
        return new this(message);
    }
}
exports.ProxyProviderNotRegisteredException = ProxyProviderNotRegisteredException;
class ProxyProviderNotResolvedException extends ProxyProviderError {
    static create(providerSymbol, propName) {
        const isDefaultProxyProvider = proxy_provider_constants_1.defaultProxyProviderTokens.has(providerSymbol);
        const providerName = typeof providerSymbol == 'string'
            ? providerSymbol
            : (providerSymbol.description ?? 'unknown');
        let message;
        if (propName) {
            message = `Cannot access the property "${propName}" on the Proxy provider`;
        }
        else {
            message = 'Cannot call the Proxy provider';
        }
        if (isDefaultProxyProvider) {
            message += ` ${providerName} because because the value for ${providerName} does not exist in the CLS. Make sure to enable the "${this.formatEnableOptionName(providerName)}" option in the enhancer options in the "ClsModule.forRoot()" method.`;
        }
        else {
            message += ` ${providerName} because is has not been resolved yet and has been registered with the "strict: true" option. Make sure to call "await cls.resolveProxyProviders()" before accessing the Proxy provider.`;
        }
        return new this(message);
    }
    static formatEnableOptionName(providerName) {
        const name = providerName.replace('CLS_', '').toLowerCase();
        return 'save' + name.charAt(0).toUpperCase() + name.slice(1);
    }
}
exports.ProxyProviderNotResolvedException = ProxyProviderNotResolvedException;
class ProxyProvidersResolutionTimeoutException extends ProxyProviderError {
    static create(timeout) {
        const message = `Proxy Providers could not be resolved within the specified timeout of ${timeout}ms, possibly due to a circular dependency. Make sure to avoid circular dependencies in your Proxy Providers.`;
        return new this(message);
    }
}
exports.ProxyProvidersResolutionTimeoutException = ProxyProvidersResolutionTimeoutException;
//# sourceMappingURL=proxy-provider.exceptions.js.map