"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.schemaDiffToSql = exports.schemaDiff = void 0;
const enum_comparer_1 = require("./comparers/enum.comparer");
const extension_comparer_1 = require("./comparers/extension.comparer");
const function_comparer_1 = require("./comparers/function.comparer");
const override_comparer_1 = require("./comparers/override.comparer");
const parameter_comparer_1 = require("./comparers/parameter.comparer");
const table_comparer_1 = require("./comparers/table.comparer");
const base_context_1 = require("./contexts/base-context");
const helpers_1 = require("./helpers");
const transformers_1 = require("./transformers");
const types_1 = require("./types");
const schemaDiff = (source, target, options = {}) => {
    const items = [
        ...(0, helpers_1.compare)(source.parameters, target.parameters, options.parameters, parameter_comparer_1.compareParameters),
        ...(0, helpers_1.compare)(source.extensions, target.extensions, options.extensions, extension_comparer_1.compareExtensions),
        ...(0, helpers_1.compare)(source.functions, target.functions, options.functions, function_comparer_1.compareFunctions),
        ...(0, helpers_1.compare)(source.enums, target.enums, options.enums, enum_comparer_1.compareEnums),
        ...(0, helpers_1.compare)(source.tables, target.tables, options.tables, table_comparer_1.compareTables),
        ...(0, helpers_1.compare)(source.overrides, target.overrides, options.overrides, override_comparer_1.compareOverrides),
    ];
    const itemMap = {
        ColumnRename: [],
        ConstraintRename: [],
        IndexRename: [],
        ExtensionDrop: [],
        ExtensionCreate: [],
        ParameterSet: [],
        ParameterReset: [],
        FunctionDrop: [],
        FunctionCreate: [],
        EnumDrop: [],
        EnumCreate: [],
        TriggerDrop: [],
        ConstraintDrop: [],
        TableDrop: [],
        ColumnDrop: [],
        ColumnAdd: [],
        ColumnAlter: [],
        TableCreate: [],
        ConstraintAdd: [],
        TriggerCreate: [],
        IndexCreate: [],
        IndexDrop: [],
        OverrideCreate: [],
        OverrideUpdate: [],
        OverrideDrop: [],
    };
    for (const item of items) {
        itemMap[item.type].push(item);
    }
    const constraintAdds = itemMap.ConstraintAdd.filter((item) => item.type === 'ConstraintAdd');
    const orderedItems = [
        ...itemMap.ExtensionCreate,
        ...itemMap.FunctionCreate,
        ...itemMap.ParameterSet,
        ...itemMap.ParameterReset,
        ...itemMap.EnumCreate,
        ...itemMap.TriggerDrop,
        ...itemMap.IndexDrop,
        ...itemMap.ConstraintDrop,
        ...itemMap.TableCreate,
        ...itemMap.ColumnAlter,
        ...itemMap.ColumnAdd,
        ...itemMap.ColumnRename,
        ...constraintAdds.filter(({ constraint }) => constraint.type === types_1.ConstraintType.PRIMARY_KEY),
        ...constraintAdds.filter(({ constraint }) => constraint.type === types_1.ConstraintType.FOREIGN_KEY),
        ...constraintAdds.filter(({ constraint }) => constraint.type === types_1.ConstraintType.UNIQUE),
        ...constraintAdds.filter(({ constraint }) => constraint.type === types_1.ConstraintType.CHECK),
        ...itemMap.ConstraintRename,
        ...itemMap.IndexCreate,
        ...itemMap.IndexRename,
        ...itemMap.TriggerCreate,
        ...itemMap.ColumnDrop,
        ...itemMap.TableDrop,
        ...itemMap.EnumDrop,
        ...itemMap.FunctionDrop,
        ...itemMap.OverrideCreate,
        ...itemMap.OverrideUpdate,
        ...itemMap.OverrideDrop,
    ];
    return {
        items: orderedItems,
        asSql: (options) => (0, exports.schemaDiffToSql)(orderedItems, options),
    };
};
exports.schemaDiff = schemaDiff;
const schemaDiffToSql = (items, options = {}) => {
    return items.flatMap((item) => asSql(item, options));
};
exports.schemaDiffToSql = schemaDiffToSql;
const asSql = (item, options) => {
    const ctx = new base_context_1.BaseContext(options);
    for (const transform of transformers_1.transformers) {
        const result = transform(ctx, item);
        if (!result) {
            continue;
        }
        return asArray(result).map((result) => result + withComments(options.comments, item));
    }
    throw new Error(`Unhandled schema diff type: ${item.type}`);
};
const withComments = (comments, item) => {
    if (!comments) {
        return '';
    }
    return ` -- ${item.reason}`;
};
const asArray = (items) => {
    if (Array.isArray(items)) {
        return items;
    }
    return [items];
};
//# sourceMappingURL=schema-diff.js.map