"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendOneShotAppRestart = sendOneShotAppRestart;
exports.createMaintenanceLoginUrl = createMaintenanceLoginUrl;
exports.signMaintenanceJwt = signMaintenanceJwt;
exports.generateMaintenanceSecret = generateMaintenanceSecret;
exports.detectPriorInstall = detectPriorInstall;
const redis_adapter_1 = require("@socket.io/redis-adapter");
const ioredis_1 = __importDefault(require("ioredis"));
const jose_1 = require("jose");
const node_crypto_1 = require("node:crypto");
const node_path_1 = require("node:path");
const socket_io_1 = require("socket.io");
const storage_core_1 = require("../cores/storage.core");
const enum_1 = require("../enum");
const config_repository_1 = require("../repositories/config.repository");
function sendOneShotAppRestart(state) {
    const server = new socket_io_1.Server();
    const { redis } = new config_repository_1.ConfigRepository().getEnv();
    const pubClient = new ioredis_1.default(redis);
    const subClient = pubClient.duplicate();
    server.adapter((0, redis_adapter_1.createAdapter)(pubClient, subClient));
    async function tryTerminate() {
        while (true) {
            try {
                const responses = await server.serverSideEmitWithAck('AppRestart', state);
                if (responses.length > 0) {
                    return;
                }
            }
            catch (error) {
                console.error(error);
                console.error('Encountered an error while telling Immich to stop.');
            }
            console.info("\nIt doesn't appear that Immich stopped, trying again in a moment.\nIf Immich is already not running, you can ignore this error.");
            await new Promise((r) => setTimeout(r, 1e3));
        }
    }
    server.emit('AppRestartV1', state, () => {
        void tryTerminate().finally(() => {
            pubClient.disconnect();
            subClient.disconnect();
        });
    });
}
async function createMaintenanceLoginUrl(baseUrl, auth, secret) {
    return `${baseUrl}/maintenance?token=${await signMaintenanceJwt(secret, auth)}`;
}
async function signMaintenanceJwt(secret, data) {
    const alg = 'HS256';
    return await new jose_1.SignJWT({ ...data })
        .setProtectedHeader({ alg })
        .setIssuedAt()
        .setExpirationTime('4h')
        .sign(new TextEncoder().encode(secret));
}
function generateMaintenanceSecret() {
    return (0, node_crypto_1.randomBytes)(64).toString('hex');
}
async function detectPriorInstall(storageRepository) {
    return {
        storage: await Promise.all(Object.values(enum_1.StorageFolder).map(async (folder) => {
            const path = storage_core_1.StorageCore.getBaseFolder(folder);
            const files = await storageRepository.readdir(path);
            const filename = (0, node_path_1.join)(storage_core_1.StorageCore.getBaseFolder(folder), '.immich');
            let readable = false, writable = false;
            try {
                await storageRepository.readFile(filename);
                readable = true;
                await storageRepository.overwriteFile(filename, Buffer.from(`${Date.now()}`));
                writable = true;
            }
            catch {
            }
            return {
                folder,
                readable,
                writable,
                files: files.filter((fn) => fn !== '.immich').length,
            };
        })),
    };
}
//# sourceMappingURL=maintenance.js.map