"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchClusterEventCoordinator = void 0;
const Mean_1 = require("./Mean");
const String_1 = require("./String");
/**
 * Centralized coordinator for BatchCluster events.
 * Handles event processing, statistics tracking, and automated responses to events.
 */
class BatchClusterEventCoordinator {
    emitter;
    options;
    onIdleLater;
    #logger;
    #tasksPerProc = new Mean_1.Mean();
    #childEndCounts = new Map();
    #internalErrorCount = 0;
    constructor(emitter, options, onIdleLater) {
        this.emitter = emitter;
        this.options = options;
        this.onIdleLater = onIdleLater;
        this.#logger = options.logger;
        this.#setupEventHandlers();
    }
    /**
     * Set up all event handlers for the BatchCluster
     */
    #setupEventHandlers() {
        this.emitter.on("childEnd", (bp, why) => this.#handleChildEnd(bp, why));
        this.emitter.on("internalError", (error) => this.#handleInternalError(error));
        this.emitter.on("noTaskData", (stdout, stderr, proc) => this.#handleNoTaskData(stdout, stderr, proc));
        this.emitter.on("startError", (error) => this.#handleStartError(error));
    }
    /**
     * Handle child process end events
     */
    #handleChildEnd(process, reason) {
        this.#tasksPerProc.push(process.taskCount);
        this.#childEndCounts.set(reason, (this.#childEndCounts.get(reason) ?? 0) + 1);
        this.onIdleLater();
    }
    /**
     * Handle internal error events
     */
    #handleInternalError(error) {
        this.#logger().error("BatchCluster: INTERNAL ERROR: " + String(error));
        this.#internalErrorCount++;
    }
    /**
     * Handle no task data events (data received without current task)
     */
    #handleNoTaskData(stdout, stderr, proc) {
        this.#logger().warn("BatchCluster: child process emitted data with no current task. Consider setting streamFlushMillis to a higher value.", {
            streamFlushMillis: this.options.streamFlushMillis,
            stdout: (0, String_1.toS)(stdout),
            stderr: (0, String_1.toS)(stderr),
            proc_pid: proc?.pid,
        });
        this.#internalErrorCount++;
    }
    /**
     * Handle start error events.
     * Logs a warning and triggers idle processing to spawn replacement processes.
     * Note: We intentionally do NOT shut down the cluster on spawn failures.
     * The minDelayBetweenSpawnMillis setting already prevents fork bombs.
     */
    #handleStartError(error) {
        this.#logger().warn("BatchCluster.onStartError(): " + String(error));
        this.onIdleLater();
    }
    /**
     * Get the mean number of tasks completed by child processes
     */
    get meanTasksPerProc() {
        const mean = this.#tasksPerProc.mean;
        return isNaN(mean) ? 0 : mean;
    }
    /**
     * Get internal error count
     */
    get internalErrorCount() {
        return this.#internalErrorCount;
    }
    /**
     * Get count of ended child processes by reason
     */
    countEndedChildProcs(reason) {
        return this.#childEndCounts.get(reason) ?? 0;
    }
    /**
     * Get all child end counts
     */
    get childEndCounts() {
        return Object.fromEntries([...this.#childEndCounts.entries()]);
    }
    /**
     * Get event statistics for monitoring
     */
    getEventStats() {
        return {
            meanTasksPerProc: this.meanTasksPerProc,
            internalErrorCount: this.internalErrorCount,
            totalChildEndEvents: [...this.#childEndCounts.values()].reduce((sum, count) => sum + count, 0),
            childEndReasons: Object.keys(this.childEndCounts),
        };
    }
    /**
     * Reset event statistics (useful for testing)
     */
    resetStats() {
        this.#tasksPerProc = new Mean_1.Mean();
        this.#childEndCounts.clear();
        this.#internalErrorCount = 0;
    }
    /**
     * Get the underlying emitter for direct event access
     */
    get events() {
        return this.emitter;
    }
}
exports.BatchClusterEventCoordinator = BatchClusterEventCoordinator;
//# sourceMappingURL=BatchClusterEventCoordinator.js.map