import child_process from "node:child_process";
import { InternalBatchProcessOptions } from "./InternalBatchProcessOptions";
import { ProcessHealthMonitor } from "./ProcessHealthMonitor";
import { Task } from "./Task";
import { WhyNotHealthy, WhyNotReady } from "./WhyNotHealthy";
/**
 * BatchProcess manages the care and feeding of a single child process.
 */
export declare class BatchProcess {
    #private;
    readonly proc: child_process.ChildProcess;
    readonly opts: InternalBatchProcessOptions;
    private readonly onIdle;
    readonly name: string;
    readonly pid: number;
    readonly start: number;
    readonly startupTaskId: number;
    failedTaskCount: number;
    /**
     * Getter for current task (required by StreamContext interface)
     */
    get currentTask(): Task<unknown> | undefined;
    /**
     * @param onIdle to be called when internal state changes (like the current
     * task is resolved, or the process exits)
     */
    constructor(proc: child_process.ChildProcess, opts: InternalBatchProcessOptions, onIdle: () => void, healthMonitor?: ProcessHealthMonitor);
    get taskCount(): number;
    get starting(): boolean;
    /**
     * @return true if `this.end()` has been requested (which may be due to the
     * child process exiting)
     */
    get ending(): boolean;
    /**
     * @return true if `this.end()` has completed running, which includes child
     * process cleanup. Note that this may return `true` and the process table may
     * still include the child pid. Call {@link BatchProcess#running()} for an authoritative
     * (but expensive!) answer.
     */
    get ended(): boolean;
    /**
     * @return true if the child process has exited (based on OS events).
     * This is now authoritative and inexpensive since it's driven by OS events
     * rather than polling.
     */
    get exited(): boolean;
    /**
     * @return a string describing why this process should be recycled, or null if
     * the process passes all health checks. Note that this doesn't include if
     * we're already busy: see {@link BatchProcess.whyNotReady} if you need to
     * know if a process can handle a new task.
     */
    get whyNotHealthy(): WhyNotHealthy | null;
    /**
     * @return true if the process doesn't need to be recycled.
     */
    get healthy(): boolean;
    /**
     * @return true iff no current task. Does not take into consideration if the
     * process has ended or should be recycled: see {@link BatchProcess.ready}.
     */
    get idle(): boolean;
    /**
     * @return a string describing why this process cannot currently handle a new
     * task, or `undefined` if this process is idle and healthy.
     */
    get whyNotReady(): WhyNotReady | null;
    /**
     * @return true iff this process is  both healthy and idle, and ready for a
     * new task.
     */
    get ready(): boolean;
    get idleMs(): number;
    /**
     * @return true if the child process is running.
     * Now event-driven first with polling fallback.
     */
    running(): boolean;
    notRunning(): boolean;
    maybeRunHealthCheck(): Task<unknown> | undefined;
    execTask<T>(task: Task<T>): boolean;
    /**
     * End this child process.
     *
     * @param gracefully Wait for any current task to be resolved or rejected
     * before shutting down the child process.
     * @param reason who called end() (used for logging)
     * @return Promise that will be resolved when the process has completed.
     * Subsequent calls to end() will ignore the parameters and return the first
     * endPromise.
     */
    end(gracefully: boolean | undefined, reason: WhyNotHealthy): Promise<void>;
}
